// -*- Pike -*-

#pike __REAL_VERSION__

inherit .___Math;

constant deg_turn = 360;
constant rad_turn = 2.0*pi;
constant gon_turn = 400;
constant str_turn = 6300;

//! This function converts between degrees, radians and gons. The
//! @[from] and @[to] arguments may be any of the follwoing
//! strings: "deg", "rad", "gon" and "str" for degrees, radians, gon
//! and streck respectivly. The output is not guaranteed to be within
//! the first turn, e.g. converting 10 radians yields almost 573
//! degrees as output.
int|float convert_angle(int|float angle, string from, string to) {
  if(from==to) return angle;
  float from_v = (float)this[from+"_turn"];
  float to_v = (float)this[to+"_turn"];
  if( from_v==0.0 )
    error("Unknown angle from type %s.\n", from);
  if( to_v==0.0 )
    error("Unknown angle to type %s.\n", from);
  if(from==to)
    return angle;
  return angle * to_v/from_v;
}

//! Represents an angle.
class Angle {

  // --- Constants

  //! The actual keeper of the angle value.
  int|float angle;

  //! The type of the angle value. Is either "deg", "rad", "gon" or "str".
  string type;

  // --- Basic general functions

  //! @decl void create()
  //! @decl void create(int|float radians)
  //! @decl void create(int|float angle, string type)
  //!
  //! If an angle object is created without arguments it
  //! will have the value 0 radians.
  void create(void|int|float angle, void|string type) {
    if(type)
      set(type, angle);
    else
      set_rad(angle);
  }

  //! Returns a copy of the object.
  Angle clone_me() {
    return Angle(angle, type);
  }

  //! Gets the value in the provided type.
  int|float get(string type) {
    switch(type) {
    case "deg": return degree();
    case "rad": return rad();
    case "gon": return gon();
    case "str": return streck();
    }
    error("Unknown type %s.\n", type);
  }

  //! Sets the angle value and type to the given value
  //! and type.
  Angle set(string type, int|float _angle) {
    switch(type) {
    case "deg": return set_degree(_angle);
    case "rad": return set_rad(_angle);
    case "gon": return set_gon(_angle);
    case "str": return set_streck(_angle);
    }
    error("Unknown type %s.\n", type);
  }

  //! Normalizes the angle to be within one turn.
  void normalize() {
    if(!type) error("No type set.\n");
    switch(type) {
    case "deg":
      angle %= deg_turn;
      break;
    case "rad":
      angle %= rad_turn;
      break;
    case "gon":
      angle %= gon_turn;
      break;
    case "str":
      angle %= str_turn;
      break;
    }
  }

  // --- Degree related functions

  //! Returns the number of degrees, including minutes and
  //! seconds as decimals.
  int|float degree() {
    return convert_angle(angle, type, "deg");
  }

  //! Returns the number of minute.
  int minute() {
    float angle = (float)degree();
    if(angle - floor(angle) == 0.0) return 0;
    return (int)floor(60.0*(angle-floor(angle)));
  }

  //! Returns the number of seconds.
  float second() {
    float angle = (float)degree();
    if(angle - floor(angle) == 0.0) return 0;
    return 3600.0*(angle-floor(60.0*angle)/60.0);
  }

  //! @decl Angle set_dms(int degrees)
  //! @decl Angle set_dms(int degrees, int minutes)
  //! @decl Angle set_dms(int degrees, int minutes, float seconds)
  //! Set degrees, minues and seconds. Returns the current angle object.
  Angle set_dms(int degrees, void|int minutes, void|float seconds) {
    set_degree( degrees + minutes/60.0 + seconds/3600.0 );
    return this;
  }

  //! Returns degrees, minutes and seconds as a string, e.g.
  //! 476'36.00".
  string format_dms() {
    return sprintf("%d%d'%.2f\"", (int)degree(), minute(), second());
  }

  //! Sets the angle to the provided degree. Alters the type to degrees.
  //! Returns the current object.
  Angle set_degree(int|float degree) {
    angle = degree;
    type = "deg";
    normalize();
    return this;
  }

  // --- Gon related functions

  //! Returns the number of gons.
  int|float gon() {
    return convert_angle(angle, type, "gon");
  }

  //! Set the angle to the provided gons. Alters the type to gons.
  //! Returns the current angle object.
  Angle set_gon(int|float gon) {
    angle = gon;
    type = "gon";
    normalize();
    return this;
  }

  // gon is formatted as %dc%d`%.2f``

  // --- Radian related functions

  //! Returns the number of radians.
  float rad() {
    return convert_angle(angle, type, "rad");
  }

  //! Set the angle to the provided radians. Alters the type to radians.
  //! Returns the current angle object.
  Angle set_rad(int|float rad) {
    angle = rad;
    type = "rad";
    normalize();
    return this;
  }

  // --- Streck related functions

  //! Returns the number of strecks.
  float|int streck() {
    return convert_angle(angle, type, "str");
  }

  //! Set the angle to the provided strecks. Alters the type to streck.
  //! Returns the current angle object.
  Angle set_streck(int|float str) {
    angle = str;
    type = "str";
    normalize();
    return this;
  }

  // --- Advanced general functions

  //! Turns the direction of the angle half a turn. Equal to
  //! @expr{add(180,"deg")@}.
  void about_face() {
    add(180, "deg");
  }

  //! Turns the direction of the angle a quarter of a turn to the right.
  //! Equal to @expr{subtract(90,"deg")@}.
  void right_face() {
    subtract(90, "deg");
  }

  //! Turns the direction of the angle a quarter of a turn to the left.
  //! Equal to @expr{add(90,"deg")@}.
  void left_face() {
    add(90, "deg");
  }

  //! Returns the sinus for the angle.
  float sin() {
    return predef::sin(rad());
  }

  //! Returns the cosinus for the angle.
  float cos() {
    return predef::cos(rad());
  }

  //! Returns the tangen for the angle.
  float tan() {
    return predef::tan(rad());
  }

  //! An angle can be casted to float, int and string.
  float|int|string cast(string to) {
    switch(to) {
    case "float": return (float)angle;
    case "int": return (int)angle;
    case "string": return (string)angle;
    }
    error( "Could not cast to %s.\n", to );
  }

  //! Returns the sum of this angle and what it is added with. If
  //! added with an angle, a new angle object is returnes.
  float|int|Angle `+ (float|int|Angle _angle) {
    if(objectp(_angle)) return Angle( angle + _angle->get(type), type );
    return angle + _angle;
  }

  //! @decl Angle add(float|int angle)
  //! @decl Angle add(float|int angle, string type)
  //! @decl Angle add(Angle angle)
  //! Adds the provided angle to the current angle. The result is
  //! normalized within 360 degrees.
  Angle add(float|int|Angle _angle, void|string _type) {
    if(objectp(_angle)) return add( _angle->angle, _type||_angle->type);
    if(!_type)
      angle += _angle;
    else
      angle += convert_angle(_angle, _type, type);
    normalize();
    return this;
  }

  //! Returns the difference between this angle and the provided value.
  //! If differenced with an angle, a new angle object is returned.
  float|int|Angle `- (float|int|Angle _angle) {
    if(objectp(_angle)) return Angle( angle - _angle->get(type), type );
    return angle - _angle;
  }

  //! @decl Angle subtract(float|int angle)
  //! @decl Angle subtract(float|int angle, string type)
  //! @decl Angle subtract(Angle angle)
  //! Subtracts the provided angle from the current angle. The result is
  //! normalized within 360 degrees.
  Angle subtract(float|int|Angle _angle, void|string _type) {
    if(objectp(_angle)) return subtract( _angle->angle, _type||_angle->type);
    if(!_type)
      angle -= _angle;
    else
      angle -= convert_angle(_angle, _type, type);
    normalize();
    return this;
  }

  //! Returns the product between this angle and the provided value.
  //! If differenced with an angle, a new angle object is returned.
  float|int|Angle `* (float|int|Angle _angle) {
    if(objectp(_angle)) return Angle( angle * _angle->get(type), type );
    return angle * _angle;
  }

  //! Returns the fraction between this angle and the provided value.
  //! If differenced with an angle, a new angle object is returned.
  float|int|Angle `/ (float|int|Angle _angle) {
    if(objectp(_angle)) return Angle( angle / _angle->get(type), type );
    return angle / _angle;
  }

  //! Returns this result of this angle modulo the provided value.
  //! If differenced with an angle, a new angle object is returned.
  float|int|Angle `% (float|int|Angle _angle) {
    if(objectp(_angle)) return Angle( angle % _angle->get(type), type );
    return angle % _angle;
  }

  //!
  int __hash() {
    return array_sscanf(sprintf("%8F", rad()),"%8c")[0];
  }

  //! Compares the unnormalized angle of two Angle objects.
  int `== (Angle _angle) {
    return _angle->rad() == rad();
  }

  //! Compares the unnormalized angle of two Angle objects.
  int `< (Angle _angle) {
    return rad() < _angle->rad();
  }

  //! Compares the unnormalized angle of two Angle objects.
  int `> (Angle _angle) {
    return rad() > _angle->rad();
  }

  string _sprintf(int t) {
    switch(t) {
    case 'O':
      switch(type) {
      case "deg": return "Angle(" + angle + ")";
      case "rad": return "Angle(" + angle + " rads)";
      case "gon": return "Angle(" + angle + " gons)";
      case "str": return "Angle(" + angle + " streck)";
      }
    case 't':
      return "Angle";
    }
    return UNDEFINED;
  }
}

//! Calculate binomial koefficient @[n] choose @[k].
//!
//! This is equvivalent to @tt{@[n]!/(@[k]!*(@[n]-@[k])!)@}.
int choose(int n, int k)
{
  if (k*2 > n) k = n-k;
  if (!k) return 1;
  if (k < 0) return 0;
  int val = 1;
  for (int i=1; i <= k; i++,n--) {
    val *= n;
    val /= i;
  }
  return val;
}

//! The 10-logarithm of @[x].
float log10(float x) { return log(x)/log(10.0); }

//! The 2-logarithm of @[x].
float log2(float x) { return log(x)/log(2.0); }


// Only available when there is Gmp.mpz
#if constant(Gmp.mpz)

//! Factorize the integer @[x]. The returned list of factors will
//! be sorted biggest to smallest factor.
//! @note
//!   This function is only available when Pike has been compiled with
//!   bignums.
array(int) factor(int x) {
  int f = Gmp.mpz(x)->small_factor();
  if(f==0)
    return ({ x });
  else
    return factor(x/f) + ({ f });
}
#endif
