// By Martin Nilsson

// Updated 2006-06-09 from http://lcweb.loc.gov/standards/iso639-2/

#pike __REAL_VERSION__

// Mapping from ISO 639-2/T code to language name.
static constant languages = ([
  "aar":"Afar",
  "abk":"Abkhazian",
  "ace":"Achinese",
  "ach":"Acoli",
  "ada":"Adangme",
  "ady":"Adyghe; Adygei",
  "afa":"Afro-Asiatic (Other)",
  "afh":"Afrihili",
  "afr":"Afrikaans",
  "ain":"Ainu",
  "aka":"Akan",
  "akk":"Akkadian",
  "ale":"Aleut",
  "alg":"Algonquian languages",
  "alt":"Southern Altai",
  "amh":"Amharic",
  "ang":"English, Old (ca.450-1100)",
  "anp":"Angika",
  "apa":"Apache languages",
  "ara":"Arabic",
  "arc":"Aramaic",
  "arg":"Aragonese",
  "arn":"Araucanian",
  "arp":"Arapaho",
  "art":"Artificial (Other)",
  "arw":"Arawak",
  "asm":"Assamese",
  "ast":"Asturian",
  "ath":"Athapascan languages",
  "aus":"Australian languages",
  "ava":"Avaric",
  "ave":"Avestan",
  "awa":"Awadhi",
  "aym":"Aymara",
  "aze":"Azerbaijani",
  "bad":"Banda",
  "bai":"Bamileke languages",
  "bak":"Bashkir",
  "bal":"Baluchi",
  "bam":"Bambara",
  "ban":"Balinese",
  "bas":"Basa",
  "bat":"Baltic (Other)",
  "bej":"Beja",
  "bel":"Belarusian",
  "bem":"Bemba",
  "ben":"Bengali",
  "ber":"Berber (Other)",
  "bho":"Bhojpuri",
  "bih":"Bihari",
  "bik":"Bikol",
  "bin":"Bini",
  "bis":"Bislama",
  "bla":"Siksika",
  "bnt":"Bantu (Other)",
  "bod":"Tibetan",
  "bos":"Bosnian",
  "bra":"Braj",
  "bre":"Breton",
  "btk":"Batak (Indonesia)",
  "bua":"Buriat",
  "bug":"Buginese",
  "bul":"Bulgarian",
  "byn":"Blin",
  "cad":"Caddo",
  "cai":"Central American Indian (Other)",
  "car":"Carib",
  "cat":"Catalan",
  "cau":"Caucasian (Other)",
  "ceb":"Cebuano",
  "cel":"Celtic (Other)",
  "ces":"Czech",
  "cha":"Chamorro",
  "chb":"Chibcha",
  "che":"Chechen",
  "chg":"Chagatai",
  "chk":"Chuukese",
  "chm":"Mari",
  "chn":"Chinook jargon",
  "cho":"Choctaw",
  "chp":"Chipewyan",
  "chr":"Cherokee",
  "chu":"Church Slavic",
  "chv":"Chuvash",
  "chy":"Cheyenne",
  "cmc":"Chamic languages",
  "cop":"Coptic",
  "cor":"Cornish",
  "cos":"Corsican",
  "cpe":"Creoles and pidgins, English based (Other)",
  "cpf":"Creoles and pidgins, French-based (Other)",
  "cpp":"Creoles and pidgins, Portuguese-based (Other)",
  "cre":"Cree",
  "crh":"Crimean Turkish",
  "crp":"Creoles and pidgins (Other)",
  "csb":"Kashubian",
  "cus":"Cushitic (Other)",
  "cym":"Welsh",
  "dak":"Dakota",
  "dan":"Danish",
  "dar":"Dargwa",
  "day":"Dayak",
  "del":"Delaware",
  "den":"Slave (Athapascan)",
  "deu":"German",
  "dgr":"Dogrib",
  "din":"Dinka",
  "div":"Divehi",
  "doi":"Dogri",
  "dra":"Dravidian (Other)",
  "dsb":"Lower Sorbian",
  "dua":"Duala",
  "dum":"Dutch, Middle (ca.1050-1350)",
  "dyu":"Dyula",
  "dzo":"Dzongkha",
  "efi":"Efik",
  "egy":"Egyptian (Ancient)",
  "eka":"Ekajuk",
  "ell":"Greek, Modern (1453-)",
  "elx":"Elamite",
  "eng":"English",
  "enm":"English, Middle (1100-1500)",
  "epo":"Esperanto",
  "est":"Estonian",
  "eus":"Basque",
  "ewe":"Ewe",
  "ewo":"Ewondo",
  "fan":"Fang",
  "fao":"Faroese",
  "fas":"Persian",
  "fat":"Fanti",
  "fij":"Fijian",
  "fil":"Filipino",
  "fin":"Finnish",
  "fiu":"Finno-Ugrian (Other)",
  "fon":"Fon",
  "fra":"French",
  "frm":"French, Middle (ca.1400-1600)",
  "fro":"French, Old (842-ca.1400)",
  "frr":"Northern Frisian",
  "frs":"Eastern Frisian",
  "fry":"Western Frisian",
  "ful":"Fulah",
  "fur":"Friulian",
  "gaa":"Ga",
  "gay":"Gayo",
  "gba":"Gbaya",
  "gem":"Germanic (Other)",
  "gez":"Geez",
  "gil":"Gilbertese",
  "gla":"Gaelic (Scots)",
  "gle":"Irish",
  "glg":"Galician",
  "glv":"Manx",
  "gmh":"German, Middle High (ca.1050-1500)",
  "goh":"German, Old High (ca.750-1050)",
  "gon":"Gondi",
  "gor":"Gorontalo",
  "got":"Gothic",
  "grb":"Grebo",
  "grc":"Greek, Ancient (to 1453)",
  "grn":"Guarani",
  "gsw":"Alemanic",
  "guj":"Gujarati",
  "gwi":"Gwichin",
  "hai":"Haida",
  "hau":"Hausa",
  "haw":"Hawaiian",
  "heb":"Hebrew",
  "her":"Herero",
  "hil":"Hiligaynon",
  "him":"Himachali",
  "hin":"Hindi",
  "hit":"Hittite",
  "hmn":"Hmong",
  "hmo":"Hiri Motu",
  "hrv":"Croatian",
  "hsb":"Upper Sorban",
  "hun":"Hungarian",
  "hup":"Hupa",
  "hye":"Armenian",
  "iba":"Iban",
  "ibo":"Igbo",
  "ido":"Ido",
  "iii":"Sichuan Yi",
  "ijo":"Ijo",
  "iku":"Inuktitut",
  "ile":"Interlingue",
  "ilo":"Iloko",
  "ina":"Interlingua (International Auxiliary Language Association)",
  "inc":"Indic (Other)",
  "ind":"Indonesian",
  "ine":"Indo-European (Other)",
  "inh":"Ingush",
  "ipk":"Inupiaq",
  "ira":"Iranian (Other)",
  "iro":"Iroquoian languages",
  "isl":"Icelandic",
  "ita":"Italian",
  "jav":"Javanese",
  "jbo":"Lojban",
  "jpn":"Japanese",
  "jpr":"Judeo-Persian",
  "kaa":"Kara-Kalpak",
  "kab":"Kabyle",
  "kac":"Kachin",
  "kal":"Kalaallisut",
  "kam":"Kamba",
  "kan":"Kannada",
  "kar":"Karen",
  "kas":"Kashmiri",
  "kat":"Georgian",
  "kau":"Kanuri",
  "kaw":"Kawi",
  "kaz":"Kazakh",
  "kbd":"Kabardian",
  "kha":"Khasi",
  "khi":"Khoisan (Other)",
  "khm":"Khmer",
  "kho":"Khotanese",
  "kik":"Kikuyu",
  "kin":"Kinyarwanda",
  "kir":"Kirghiz",
  "kmb":"Kimbundu",
  "kok":"Konkani",
  "kom":"Komi",
  "kon":"Kongo",
  "kor":"Korean",
  "kos":"Kosraean",
  "kpe":"Kpelle",
  "krc":"Karachay-Balkar",
  "krl":"Karelian",
  "kro":"Kru",
  "kru":"Kurukh",
  "kum":"Kumyk",
  "kur":"Kurdish",
  "kut":"Kutenai",
  "lad":"Ladino",
  "lah":"Lahnda",
  "lam":"Lamba",
  "lao":"Lao",
  "lat":"Latin",
  "lav":"Latvian",
  "lez":"Lezghian",
  "lim":"Limburgish",
  "lin":"Lingala",
  "lit":"Lithuanian",
  "lol":"Mongo",
  "loz":"Lozi",
  "ltz":"Letzeburgesch",
  "lua":"Luba-Lulua",
  "lub":"Luba-Katanga",
  "lug":"Ganda",
  "lui":"Luiseno",
  "lun":"Lunda",
  "luo":"Luo (Kenya and Tanzania)",
  "lus":"lushai",
  "mad":"Madurese",
  "mag":"Magahi",
  "mah":"Marshall",
  "mai":"Maithili",
  "mak":"Makasar",
  "mal":"Malayalam",
  "man":"Mandingo",
  "map":"Austronesian (Other)",
  "mar":"Marathi",
  "mas":"Masai",
  "mdf":"Moksha",
  "mdr":"Mandar",
  "men":"Mende",
  "mga":"Irish, Middle (900-1200)",
  "mic":"Micmac",
  "min":"Minangkabau",
  "mis":"Miscellaneous languages",
  "mkd":"Macedonian",
  "mkh":"Mon-Khmer (Other)",
  "mlg":"Malagasy",
  "mlt":"Maltese",
  "mnc":"Manchu",
  "mni":"Manipuri",
  "mno":"Manobo languages",
  "moh":"Mohawk",
  "mol":"Moldavian",
  "mon":"Mongolian",
  "mos":"Mossi",
  "mri":"Maori",
  "msa":"Malay",
  "mul":"Multiple languages",
  "mun":"Munda languages",
  "mus":"Creek",
  "mwl":"Mirandese",
  "mwr":"Marwari",
  "mya":"Burmese",
  "myn":"Mayan languages",
  "myv":"Erzya",
  "nah":"Nahuatl",
  "nai":"North American Indian",
  "nap":"Neapolitan",
  "nau":"Nauru",
  "nav":"Navajo",
  "nbl":"Ndebele, South",
  "nde":"Ndebele, North",
  "ndo":"Ndonga",
  "nds":"Low German; Low Saxon",
  "nep":"Nepali",
  "new":"Newari",
  "nia":"Nias",
  "nic":"Niger-Kordofanian (Other)",
  "niu":"Niuean",
  "nld":"Dutch",
  "nno":"Norwegian Nynorsk",
  "nob":"Norwegian Bokml",
  "non":"Norse, Old",
  "nor":"Norwegian",
  "nso":"Northern Sotho",
  "nub":"Nubian languages",
  "nqo":"N'Ko",
  "nwc":"Classical Newari",
  "nya":"Chichewa",
  "nym":"Nyamwezi",
  "nyn":"Nyankole",
  "nyo":"Nyoro",
  "nzi":"Nzima",
  "oci":"Occitan (post 1500); Provenal",
  "oji":"Ojibwa",
  "ori":"Oriya",
  "orm":"Oromo",
  "osa":"Osage",
  "oss":"Ossetian; Ossetic",
  "ota":"Turkish, Ottoman (1500-1928)",
  "oto":"Otomian languages",
  "paa":"Papuan (Other)",
  "pag":"Pangasinan",
  "pal":"Pahlavi",
  "pam":"Pampanga",
  "pan":"Panjabi",
  "pap":"Papiamento",
  "pau":"Palauan",
  "peo":"Persian, Old (ca.600-400 B.C.)",
  "phi":"Philippine (Other)",
  "pli":"Pali",
  "pol":"Polish",
  "pon":"Pohnpeian",
  "por":"Portuguese",
  "pra":"Prakrit languages",
  "pro":"Provenal, Old (to 1500)",
  "pus":"Pushto",
// qaa-qtz is Reserved for local use
  "que":"Quechua",
  "raj":"Rajasthani",
  "rap":"Rapanui",
  "rar":"Rarotongan",
  "roa":"Romance (Other)",
  "rom":"Romany",
  "ron":"Romanian",
  "run":"Rundi",
  "rup":"Aromanian",
  "rus":"Russian",
  "sad":"Sandawe",
  "sag":"Sango",
  "sah":"Yakut",
  "sai":"South American Indian (Other)",
  "sal":"Salishan languages",
  "sam":"Samaritan Aramaic",
  "san":"Sanskrit",
  "sas":"Sasak",
  "sat":"Santali",
  "scn":"Sicilian",
  "sco":"Scots",
  "sel":"Selkup",
  "sem":"Semitic (Other)",
  "sga":"Irish, Old (to 900)",
  "sgn":"Sign Languages",
  "shn":"Shan",
  "sid":"Sidamo",
  "sin":"Sinhalese",
  "sio":"Siouan languages",
  "sit":"Sino-Tibetan (Other)",
  "sk":"Sknska",
  "sla":"Slavic (Other)",
  "slk":"Slovak",
  "slv":"Slovenian",
  "sma":"Southern Sami",
  "sme":"Northern Sami",
  "smi":"Sami languages (Other)",
  "smj":"Lule Sami",
  "smn":"Inari Sami",
  "smo":"Samoan",
  "sms":"Skolt Sami",
  "sna":"Shona",
  "snd":"Sindhi",
  "snk":"Soninke",
  "sog":"Sogdian",
  "som":"Somali",
  "son":"Songhai",
  "sot":"Sotho, Southern",
  "spa":"Spanish",
  "sqi":"Albanian",
  "srd":"Sardinian",
  "srp":"Serbian",
  "srr":"Serer",
  "ssa":"Nilo-Saharan (Other)",
  "ssw":"Swati",
  "suk":"Sukuma",
  "sun":"Sundanese",
  "sus":"Susu",
  "sux":"Sumerian",
  "swa":"Swahili",
  "swe":"Swedish",
  "syr":"Syriac",
  "tah":"Tahitian",
  "tai":"Tai (Other)",
  "tam":"Tamil",
  "tat":"Tatar",
  "tel":"Telugu",
  "tem":"Timne",
  "ter":"Tereno",
  "tet":"Tetum",
  "tgk":"Tajik",
  "tgl":"Tagalog",
  "tha":"Thai",
  "tig":"Tigre",
  "tir":"Tigrinya",
  "tiv":"Tiv",
  "tkl":"Tokelau",
  "tlh":"Klingon",
  "tli":"Tlingit",
  "tmh":"Tamashek",
  "tog":"Tonga (Nyasa)",
  "ton":"Tonga (Tonga Islands)",
  "tpi":"Tok Pisin",
  "tsi":"Tsimshian",
  "tsn":"Tswana",
  "tso":"Tsonga",
  "tuk":"Turkmen",
  "tum":"Tumbuka",
  "tup":"Tupi",
  "tur":"Turkish",
  "tut":"Altaic (Other)",
  "tvl":"Tuvalu",
  "twi":"Twi",
  "tyv":"Tuvinian",
  "udm":"Udmurt",
  "uga":"Ugaritic",
  "uig":"Uighur",
  "ukr":"Ukrainian",
  "umb":"Umbundu",
  "und":"Undetermined",
  "urd":"Urdu",
  "uzb":"Uzbek",
  "vai":"Vai",
  "ven":"Venda",
  "vie":"Vietnamese",
  "vol":"Volapk",
  "vot":"Votic",
  "wak":"Wakashan languages",
  "wal":"Walamo",
  "war":"Waray",
  "was":"Washo",
  "wen":"Sorbian languages",
  "wln":"Walloon",
  "wol":"Wolof",
  "xal":"Kalmyk",
  "xho":"Xhosa",
  "yao":"Yao",
  "yap":"Yapese",
  "yid":"Yiddish",
  "yor":"Yoruba",
  "ypk":"Yupik languages",
  "zap":"Zapotec",
  "zen":"Zenaga",
  "zha":"Zhuang",
  "zho":"Chinese",
  "znd":"Zande",
  "zul":"Zulu",
  "zun":"Zuni",
  "zxx":"No linguistic content",
]);

// Mapping from ISO 639-2/B to ISO 639-2/T
static constant b_to_t = ([
  "alb":"sqi",
  "arm":"hye",
  "baq":"eus",
  "bur":"mya",
  "chi":"zho",
  "cze":"ces",
  "dut":"nld",
  "fre":"fra",
  "geo":"kat",
  "ger":"deu",
  "gre":"ell",
  "ice":"isl",
  "mac":"mkd",
  "mao":"mri",
  "may":"msa",
  "per":"fas",
  "rum":"ron",
  "scc":"srp",
  "scr":"hrv",
  "slo":"slk",
  "tib":"bod",
  "wel":"cym",
]);

//! Look up the language name given an ISO 639-2 code
//! in lower case. It will first be looked up in the
//! ISO 639-2/T table and then in ISO 639-2/B if the
//! first lookup failed. Returns zero typed zero on
//! failure.
string get_language(string code) {
  return languages[code] || languages[b_to_t[code]];
}

//! Look up the language name given an ISO 639-2/T code
//! in lower case. Returns zero typed zero on failure.
string get_language_t(string code) {
  return languages[code];
}

//! Look up the language name given an ISO 639-2/B code
//! in lower case. Returns zero typed zero on failure.
string get_language_b(string code) {
  return languages[b_to_t[code]];
}

//! Return a mapping from ISO 639-2/T + ISO 639-2/B codes
//! to language names.
mapping(string:string) list_languages() {
  return languages + map(b_to_t, lambda(string in) {
				   return languages[in]; } );
}

//! Return a mapping from ISO 639-2/T codes to language
//! names.
mapping(string:string) list_languages_t() {
  return languages + ([]);
}

//! Return a mapping from ISO 639-2/B codes to language
//! names.
mapping (string:string) list_languages_b() {
  array symbols = values(b_to_t);
  return list_languages() - mkmapping( symbols, symbols );
}

//! Converts an ISO 639-2/B code to an ISO 639-2/T code.
string convert_b_to_t(string code) {
  string res = b_to_t[code];
  if(res) return res;
  if(languages[code]) return code;
  return 0;
}

//! Converts an ISO 639-2/T code to an ISO 639-2/B code.
string convert_t_to_b(string code) {
  string res = search(b_to_t, code);
  if(res) return res;
  if(languages[code]) return code;
  return 0;
}

// Mapping from ISO 639-1 code to ISO 639-2/T code.
static constant conversion = ([
  "aa":"aar",
  "ab":"abk",
  "ae":"ave",
  "af":"afr",
  "am":"amh",
  "an":"arg",
  "ar":"ara",
  "as":"asm",
  "ay":"aym",
  "az":"aze",
  "ba":"bak",
  "be":"bel",
  "bg":"bul",
  "bh":"bih",
  "bi":"bis",
  "bn":"ben",
  "bo":"bod",
  "br":"bre",
  "bs":"bos",
  "ca":"cat",
  "ce":"che",
  "ch":"cha",
  "co":"cos",
  "cs":"ces",
  "cu":"chu",
  "cv":"chv",
  "cy":"cym",
  "da":"dan",
  "de":"deu",
  "dv":"div",
  "dz":"dzo",
  "el":"ell",
  "en":"eng",
  "eo":"epo",
  "es":"spa",
  "et":"est",
  "eu":"eus",
  "fa":"fas",
  "fi":"fin",
  "fj":"fij",
  "fo":"fao",
  "fr":"fra",
  "fy":"fry",
  "ga":"gle",
  "gd":"gla",
  "gl":"glg",
  "gn":"grn",
  "gu":"guj",
  "gv":"glv",
  "he":"heb",
  "hi":"hin",
  "ho":"hmo",
  "hr":"hrv",
  "hu":"hun",
  "hy":"hye",
  "hz":"her",
  "ia":"ina",
  "id":"ind",
  "ie":"ile",
  "ii":"iii",
  "ik":"ipk",
  "io":"ido",
  "is":"isl",
  "it":"ita",
  "iu":"iku",
  "ja":"jpn",
  "jv":"jav",
  "ka":"kat",
  "ki":"kik",
  "kk":"kaz",
  "kl":"kal",
  "km":"khm",
  "kn":"kan",
  "ko":"kor",
  "ks":"kas",
  "ku":"kur",
  "kv":"kom",
  "kw":"cor",
  "ky":"kir",
  "la":"lat",
  "lb":"ltz",
  "li":"lim",
  "ln":"lin",
  "lo":"lao",
  "lt":"lit",
  "lv":"lav",
  "mg":"mlg",
  "mh":"mah",
  "mi":"mri",
  "mk":"mkd",
  "ml":"mal",
  "mn":"mon",
  "mo":"mol",
  "mr":"mar",
  "ms":"msa",
  "mt":"mlt",
  "my":"mya",
  "na":"nau",
  "nb":"nob",
  "nd":"nde",
  "ne":"nep",
  "ng":"ndo",
  "nl":"nld",
  "nn":"nno",
  "no":"nor",
  "nr":"nbl",
  "nv":"nav",
  "ny":"nya",
  "oc":"oci",
  "om":"orm",
  "or":"ori",
  "os":"oss",
  "pa":"pan",
  "pi":"pli",
  "pl":"pol",
  "ps":"pus",
  "pt":"por",
  "qu":"que",
  "rn":"run",
  "ro":"ron",
  "ru":"rus",
  "rw":"kin",
  "sa":"san",
  "sc":"srd",
  "sd":"snd",
  "se":"sme",
  "sg":"sag",
  "si":"sin",
  "sk":"slk",
  "sl":"slv",
  "sm":"smo",
  "sn":"sna",
  "so":"som",
  "sq":"sqi",
  "sr":"srp",
  "ss":"ssw",
  "st":"sot",
  "su":"sun",
  "sv":"swe",
  "sw":"swa",
  "ta":"tam",
  "te":"tel",
  "tg":"tgk",
  "th":"tha",
  "tk":"tuk",
  "tl":"tgl",
  "tn":"tsn",
  "tr":"tur",
  "ts":"tso",
  "tt":"tat",
  "tw":"twi",
  "ty":"tah",
  "ug":"uig",
  "uk":"ukr",
  "ur":"urd",
  "uz":"uzb",
  "vi":"vie",
  "vo":"vol",
  "wa":"wln",
  "wo":"wol",
  "xh":"xho",
  "yi":"yid",
  "za":"zha",
  "zh":"zho",
  "zu":"zul"
]);

//! Look up the ISO 639-2/T code given an ISO 639-1 code
//! in lower case.
string map_639_1(string code) {
  return conversion[code];
}

//! Look up the ISO 639-1 code given an ISO 639-2/T code
//! in lower case.
string map_to_639_1(string code) {
  return search(conversion, code);
}

//! Return a mapping from ISO 639-1 code to ISO 639-2/T code.
mapping(string:string) list_639_1() {
  return conversion + ([]);
}
