#!/usr/bin/perl -T

#   Copyright (C) 2002 Mindspeed Technologies
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
#


=head1 NAME

cvscop.pl - CVS policy enforcement

=head1 VERSION

0.6 (give or take)

=head1 SYNOPSIS

B<cvscop.pl> -h|--help

B<cvscop.pl> --man

B<cvscop.pl> [-v | --verbose] mode [--sql] [-r | --cvsroot I<path>]
          [-u | --user | --cvsuser I<name>] [--module I<module>]
          [--svv I<${sVv}>]

=cut

# the rest of the docs are at the end of the file


# TODO: config file from the command line
# TODO: put Tagtree.pm into cvscop instead of a library

use DBI;
use Getopt::Long;   # command line options

# ---------------------
# CONFIGURATION SECTION
# ---------------------

# the name of the XML configuration file

my $configfile = 'cvscop-config.xml';   # absolute or relative location


# opt_verbose is overridden by the '-v' command like
my $opt_verbose = '';   # no verbosity by default

# opt_sql is overridden by either the command line or the XML file.
my $opt_sql     = 0;    # do not log to sql by default

# -------------------------
# END CONFIGURATION SECTION
# -------------------------
#
# Nothing beyond here should need to be edited for a particular installation.
# All other configuration is done in the XML config file

my $opt_help = 0;       # display usage statement
my $opt_man  = 0;       # attempt to do a manpage

## Handle command line options

# allow concatenated single-character command line options:
Getopt::Long::Configure ("bundling");

# set the defaults

my %attr        = ();   # the attributes themselves

# Attributes and which modes use them
#
#               Modes->        commitinfo  verifymsg  loginfo  taginfo

$attr{'action'}    = '';    #                                    auto   #
$attr{'arg'}       = '';    #     auto       auto       auto     auto   #
$attr{'cvsroot'}   = '';    #     arg        arg        arg      arg    #
$attr{'cvsuser'}   = '';    #     arg        arg        arg      arg    #
$attr{'path'}      = '';    #     auto                           auto   #
$attr{'module'}    = '';    #    derived                        derived #
$attr{'message'}   = '';    #                auto       auto            #
$attr{'svv'}       = '';    #                           arg             #
$attr{'tag'}       = '';    #                                    auto   #
$attr{'tempfile'}  = '';    #                auto                       #
#    {'ary_files'}          #     auto                  arg      auto   #
#    {'ary_version'}        #                           arg      auto   #
#    {'ary_oldver'}         #                           arg             #

# gather the options themselves

if (join(" ", @ARGV) =~ / echo | -v /) {
    print "Args: " . join(" ", @ARGV) . "\n";
}

$usage = <<EOF;
Usage:  $0 --help
            ( display this page )
        $0 --man
            ( attempt to display the manual page )
        $0 [-v|--verbose] mode [--sql]
           [-r|--cvsroot path] [-u|--user|--cvsuser name]
           [special options -- see manual]
Options:
    --verbose|-v            print out all steps
    --cvsroot|-r            
    --cvsuser|--user|-u     
Modes:
    echo | taginfo | verifymsg | loginfo | commitinfo
EOF

die $usage unless $ARGV[0];     # useful for commandline testing
GetOptions (
                'help|h'           => \$opt_help,
                'man'              => \$opt_man,
                'verbose|v'        => \$opt_verbose,
                'sql'              => \$opt_sql, # should have a config option
                'arg=s'            => \$attr{'arg'},
                'tag=s'            => \$attr{'tag'},
                'action=s'         => \$attr{'action'},
                'cvsroot|r=s'      => \$attr{'cvsroot'},
                'cvsuser|user|u=s' => \$attr{'cvsuser'},
                'module=s'         => \$attr{'module'},
                'message=s'        => \$attr{'message'},
                'sVv|svv=s'        => \$attr{'svv'},         # ${sVv}
            );

if ($opt_man) {

    # this is _really_ hacky, but probably a lot more convenient than finding
    # pod2html and piping it into nroff and a pager.

    my $pager = '';
    $ENV{PATH} =~ /^(.*)$/;
    local $ENV{PATH} = $1;
    $ENV{'PAGER'} =~ /^(.*)$/;  # taint badness
    for ($1, 'less', 'more') {
        $pager = `which $_`;
        $pager =~ s/^no\s.*$|^.*not\sfound\s*$//sg;
        last if $pager;
    }

    my $nroff   = `which nroff`;
    $nroff      =~ s/^no\s.*$|^.*not\sfound\s*$//sg;
    my $pod2man = `which pod2man`;
    $pod2man    =~ s/^no\s.*$|^.*not\sfound\s*$//sg;

    unless ($pager and $nroff and $pod2man) {
        warn "Cannot find a pager (more, less, etc.)!\n" unless $pager;
        warn "Cannot find nroff!\n" unless $nroff;
        warn "Cannot find pod2man!\n" unless $pod2man;
        die "Unable to display man page";
    }

    my $line = "$pod2man $0 | $nroff -man | $pager";
    $line =~ s/\s+/ /sg;
    $line =~ /^(.*)$/;

    print "$line\n";

    exec $1;

    die;

}



die $usage if $opt_help;

$attr{'arg'}       = shift unless $attr{'arg'};

my $db          = 'cvscop';
my $dbuser      = 'cvscop';
my $dbpasswd    = 'cvscop';
my $dbhost      = 'localhost';
my $dbport      = '8090';

if ($configfile !~ /^\//) {
    my $thisdir = $0;
    $thisdir    =~ s/^(.*\/)[^\/]*$/$1/g;
    $configfile = $thisdir . $configfile;   # prepend cwd
}

print "Reading config file $configfile\n" if $opt_verbose;
$configtree = new Tagtree;

if (-f $configfile) {
    open FILE, $configfile or die "Cannot open file $configfile\n";
    $the_input  = join("\n", <FILE>);
    close FILE;
} else {
    die "Cannot open file $configfile\n";
}


$configtree->parse($the_input);


# Now to parse out some of the command line args

for ($attr{'svv'}) {
    if (/\%\{/) {$_  = ''};     # undef it if it didn't get parsed by CVS

    # This string is a pain to parse, and this probably isn't 100% accurate.

    if (/^\s*(\S+)\s+(.*)$/) {    # capture
        $attr{'path'} = $1;
        my $split_this = $2;
        for (qw(ary_files ary_oldver ary_version)) { $attr{$_} = [] };
        while ($split_this =~ s/^\s*([^,]+,[^,]+,\S+)//) {
            for ($1) {
               if  (/^([^,]+),([^,]+),([^,]+)/) {
                    # XXX these arrays _might_ be better off as two hashes
                    #     keyed by filename.
                    push @{$attr{'ary_files'}},  $1;
                    push @{$attr{'ary_oldver'}}, $3;
                    push @{$attr{'ary_version'}}, $3;
                } else {
                    warn "(--svv) Cannot parse:\t$_\n";
                }
            }
        }
    }
}

SELECT: for ($attr{'arg'}) {

    /^taginfo$/ && do {
        $attr{'tag'}    = shift(@ARGV);
        $attr{'action'} = shift(@ARGV);
        $attr{'path'}   = shift(@ARGV);
        while ($ARGV[1]) { # two args available
            push @{$attr{'ary_files'}}, shift(@ARGV);
            push @{$attr{'ary_version'}}, shift(@ARGV);
        }
    };

    /^verifymsg$/ && do {
        # pop() is a better idea since the attribs are all appended
        $attr{'tempfile'} = pop unless $attr{'tempfile'};
        if (-f $attr{'tempfile'}) {
            open FILE, $attr{'tempfile'};
            $attr{'message'} = join "\n", <FILE>;
            close FILE;
        }

    };

    /^loginfo$/ && do {
        $attr{'message'} = join("\n", <STDIN>);

    };

    /^commitinfo$/ && do {
        $attr{'path'}   = shift @ARGV;
        while ($ARGV[0]) { push @{$attr{'ary_files'}}, shift @ARGV };

    };

    # work module, path, and cvsroot together

    if ($attr{'path'}) {
        if ($attr{'cvsroot'}) {

            # pull the cvs root from the path

            if ($attr{'path'} !~ s/^\/*$attr{'cvsroot'}\/*//g) {
            }

            # pull the module from the path

            if ($attr{'module'}) {
                $attr{'path'} =~ s/^\/*$attr{'module'}\/*//sg;
            } else {
                # just use the first directory in the path
                if ($attr{'path'} =~ s/^\/*([^\/]+)\/+//g) {
                    $attr{'module'} = $1;
                } elsif ($attr{'path'} !~ /.\/./) {
                    $attr{'module'} = $attr{'path'};
                    $attr{'path'} = undef;
                } else {
                    warn "Warning: could not get a module name from path.\n"
                        if $opt_verbose;
                }
            }
        }
        # for now we'll skip trying to read the path if CVSROOT isn't set.
    }

    (/^echo$/ || $opt_verbose) && do {

        print "Uncaptured args: " . join(" ", @ARGV) . "\n" if $ARGV[0];

        print $attr{'arg'} . "\nCaptured args are:\n";
        for (sort keys %attr) {
            if (/^ary_/) {
                print "\t$_\n";
                for (@{$attr{$_}}) {
                    print "\t\t$_\n";
                }
            } else {
                print "\t$_:\t" . $attr{$_} . "\n" if $attr{$_}
            }
        }
        print"\nstdin is:\n";
        print join("\n", <STDIN>);

        print "\n(eof)\n\n";

    };

}



#
# ------------------
#  XML FILE SECTION
# ------------------
#
# The file was physically read earlier, but this is where it dives down into
# the tree for configuration info and rulesets.  If an operation fails due to
# policy, this is where it happens.

print "Building XML tree...\n" if $opt_verbose;

if ($configtree->{nodes}[0]->{tagname} !~ /^cvscop$/i) {
    print "ERROR: root tag in config file is not <CVSCOP>\n";
    exit 1;
} else {

    for $uppertag ( @{ $configtree->{nodes}[0]->{children} } ) {

        OUTERNODES: for ($uppertag->{tagname}) {

            /^config$/i     && do {

                print "Reading config tags...\n" if $opt_verbose;

                for ( @{ $uppertag->{children} } ) {
                    &config_tag($_);    # read config tags
                }
                last OUTERNODES;
            };

            /^ruleset$/i    && do {

                print "Reading ruleset...\n" if $opt_verbose;

                for ( @{ $uppertag->{children} } ) {
                    &ruleset_check($_);    # test against rules
                }
                last OUTERNODES;

            };

            # the default:

            print "ERROR: Unknown tag type <" . $_ . "> in configuration file\n";

        }
    }

}




#
# -----------------------
#  POST XML-FILE SECTION
# -----------------------
#
# After this point, all of the config-file reading and policing has been done.
# Now the final operations happen
#



if (($attr{'arg'} =~ /^loginfo$/i) && $opt_sql) {

    print "Recording loginfo.\nDatabase: $db\nUser: $dbuser\n" if $opt_verbose;

    # Gether log references
    #
    # TODO: be able to gather from more than one log ref line

    for (split("\n",$attr{'message'})) {
        if (/^(LOG|MERGE):\s*(\S+.*)$/) {
            $logtype = $1;
            $logref  = $2;
        }
    }

    my $message_cooked = $attr{'message'};
    for ($message_cooked) {
        s/\r\n/\n/;
        s/^.*Log Message:[^\n]*\n(.*)$/$1/s;
    }

    $dbh = DBI->connect("DBI:mysql:$db:$dbhost:$dbport",$dbuser,$dbpasswd)
        or die("cannot open database");


    # changes table
    $sql = "INSERT INTO cvscop_changes SET id = NULL "
         . ", user ="         . $dbh->quote($attr{'cvsuser'})
         . ", module ="       . $dbh->quote($attr{'module'})
         . ", message_raw ="  . $dbh->quote($attr{'message'})
         . ", message_text =" . $dbh->quote($message_cooked)
         . ", root ="         . $dbh->quote($attr{'cvsroot'})
         . ", ismerge ="      . ( ($logtype =~ /LOG/) ? 0 : 1 );
    $sth = $dbh->prepare($sql) or die("bad sql");
    $sth->execute or die("execute failed");
    $newid = $dbh->{'mysql_insertid'};
    print "\n$sql\ninsertid = $newid \n" if $opt_verbose;

    # bugrefs table
    @refs = split(/[, ]/, "$logref") ;
    foreach $r (@refs) {
        $sql = "INSERT INTO cvscop_bugrefs SET id = $newid , ref ="
            . $dbh->quote($r)
            . ", ismerge =" . ( ($logtype =~ /LOG/) ? 0 : 1 );
        print "sql is $sql\n" if $opt_verbose;
        $sth = $dbh->prepare($sql) or die("bad sql");
        $sth->execute or die("execute failed");
    }

    # files table
    for (my $i = 0; $i < scalar(@{$attr{'ary_files'}}); $i++) {
        $sql = "INSERT INTO cvscop_files SET id = $newid "
            . ", name =" . $dbh->quote(@{$attr{'ary_files'}}[$i])
            . ", orev =" . $dbh->quote(@{$attr{'ary_oldver'}}[$i])
            . ", nrev =" . $dbh->quote(@{$attr{'ary_version'}}[$i]);
        print "sql is $sql\n" if $opt_verbose;
        $sth = $dbh->prepare($sql) or die("bad sql");
        $sth->execute or die("execute failed");
    }

}







# -----------
#  FUNCTIONS
# -----------

sub config_tag() {

    my $me = shift;

    for ($me->{'attributes'}->{name}) {

        $db         = $me->{'attributes'}->{value} if /^db$/i;
        $dbuser     = $me->{'attributes'}->{value} if /^dbuser$/i;
        $dbpasswd   = $me->{'attributes'}->{value} if /^dbpasswd$/i;
        $dbhost     = $me->{'attributes'}->{value} if /^dbhost$/i;
        $dbport     = $me->{'attributes'}->{value} if /^dbport$/i;

        if (/sql|log/i) {
            for ($me->{'attributes'}->{value}) {
                $opt_sql = 1 if /^enabled$|^on$|^active$|^yes$/i;
                $opt_sql = 0 if /^disabled$|^off$|^inactive$|^no$/i;
            }
        }

    }

    # there is no reason to nest <param> tags inside each other, but just in
    # case someone wants to do that (yuck!), here we are:

    for ( @{ $me->{children} } ) {
        &config_tag($_) unless $_->{tagname} !~ /^param$/i;    # recurse
    }

}





sub ruleset_check() {

    my $me = shift;

    DENY: for ($me->{tagname}) {

        # if we get a <deny>, we exit with a fail status

        /^deny$/i && do {

            print "\n+" . ("-" x 60) . "\n";

            if ($me->{text}) {
                (my $texttemp = $me->{text}) =~ s/ *\\n */\n/g;
                print $texttemp;
                print "\n";
            } else {
                print "Permission Denied.\n";
            }
            print "+" . ("-" x 60) . "\n\n";
            exit 1;
        };

        # XXX run external programs

    # a <run> tag example:
    #
    # <run each="some_attr" errcode="yes"> <!-- run once for each some_attr -->
    #   /path/to/program --arg=$some_attr --another_arg=$another_attr
    # </run>
    #
    #

        /^run$/i && do {

            my $run_each = '';
            my $run_err  = 0;

            # TODO: make the 'each' arguments and the match field use a common
            #       lookup to match internal name to external name

            if ( defined $attr{$me->{'attributes'}->{'each'}}){
                if ($me->{'attributes'}->{'each'} =~ /^ary_/) {
                    for (@{$attr{$me->{'attributes'}->{'each'}}}) {
                        $run_err |= &run_ext($me, $_);
                    }
                } else {
                    $run_err = &run_ext($me, $attr{$me->{'attributes'}->{'each'}});
                }
            } elsif (not defined $me->{'attributes'}->{'each'}) {
                $run_err = &run_ext($me, undef);
            } else {
                die "Bad <run> tag in $configfile!\n";
            }

            if ($me->{'attributes'}->{'errcode'} && ($run_err != 0)) {
                print "\n+" . ("-" x 60) . "\n";
                print "Failed external test.";
                print "\n+" . ("-" x 60) . "\n";
                exit 1;
            }

        };

        # <match>: run regexp against a variable

        /^match$/i && do {

            print "Matching against "  . $me->{'attributes'}->{field}
                . ":\t" . $me->{text}
                . " -- " if $opt_verbose;

            my $matched  = 0;   # OR assumes fail
            if ($me->{'attributes'}->{bool} =~ /^and$/i) {
                $matched = 1;   # AND assumes pass
            }

            # What text are we looking at?

            my @matchthis   = ();
            for ($me->{'attributes'}->{field}) {

                if (defined $attr{$_}) { # There is an attribute with this name
                    push @matchthis, $attr{$_};
                } else {            # no literal match; try these:
                   /^mode$/i     && { push @matchthis, $attr{'arg'}};
                   /^filename$/i && { push @matchthis, @{$attr{'ary_files'}}};
                   /^user$/i     && { push @matchthis, $attr{'cvsuser'}};
                }

                print join(" ",@matchthis) . "\n" if $opt_verbose;
            }

            # What is our regular expression?

            # TODO: @c_regex should use hashes rather than putting dissimilar
            #       data into a 2-D array

            my  @c_regex   = ();    # an array of arrays: 
                                    #   compiled regexes
                                    #   ! (NOT) characters

            my  $raw_regex =  $me->{text};

            # is it properly delimited?
            if ($raw_regex =~ /^\s*[!]*\/.*\/\w*\s*$/) {   # delimited regex


                # munch the delimited regexes
                while ($raw_regex =~ s/^\s*([!]*)\/([^\/]*[^\\])\/(\w*)\s*//s) {
                    my $cloister  =  $3;
                    $cloister     =~ s/(?![imsx]).//sg;
                    push @c_regex, [ ( qr/(?$cloister)$2/, $1 ) ];#compile regex
                }

            } else {

                print "Warning: cvscop config file contains non-regex "
                    . "patterns!\n";

                print "$raw_regex\n\n" if $opt_verbose;
                # no delimiters? raw text, then

                $raw_regex =~ s/^\s+//s;
                $raw_regex =~ s/\s+$//s;

                # we push an uncompiled string into the compiled regex array.
                # This means it will be taken literally instead.

                push @c_regex, ($raw_regex);

            }

            for my $regex (@c_regex) {
                # match against @matchthis
                for (@matchthis) {

                    # here is where we deal with all of the ANDs ORs and NOTs

                    if ($me->{'attributes'}->{bool} =~ /^and$/i) {    # AND
                        if ($regex->[1] =~ /^$/) {                    # 
                            if ($_ !~ /$regex->[0]/) {                # !~
                                $matched = 0;                         # fail
                            }
                        } else {                                      # NOT
                            if ($_ =~ /$regex->[0]/) {                # =~
                                $matched = 0;                         # fail
                            }
                        }
                    } else {                                          # OR
                        if ($regex->[1] =~ /^$/) {                    #
                            if ($_ =~ /$regex->[0]/) {                # =~
                                $matched = 1;                         # pass
                            }
                        } else {                                      # NOT
                            if ($_ !~ /$regex->[0]/) {                # !~
                                $matched = 1;                         # pass
                            }
                        }
                    }
                }
            }
            if ($matched == 1) {
                # recurse for each child node
                for (@{$me->{children}}) { &ruleset_check($_) };
            }
        };
    }
}






sub run_ext {

    my $me = shift;
    my $returnval = 0;
    (my $thisarg = shift) =~ s/([;'"])/\\$1/sg;
    my $cmdline = $me->{'text'};
    my $attrtemp = '';

    for ($cmdline) {
        s/\$_/$thisarg/g;
        for my $thiskey (keys %attr) {
            next if /^ary_/; # array attribs can be sent to $_ only
            ($attrtemp = $attr{$thiskey}) =~ s/([;'"])/\\$1/sg;
            s/\$$thiskey/$attrtemp/g;
        }
    }

    $cmdline =~ s/((?!>\\)[;\{\}\(\)\&\$])/\\$1/g;  # clean it up
    $cmdline =~ /^(.*)$/;  # taint cheat

    # now we run it

    local $ENV{PATH} = '';

    print `$1`; # shell out to the command

    $returnval |= $?;
    return $returnval;
}


=head1 DESCRIPTION

B<cvscop> is a CVS policy program, launched by the CVS administrative
files (B<commitinfo>, B<loginfo>, B<taginfo>, and B<verifymsg>).  It can deny
CVS actions or launch external scripts based on its configured rule set.

The output of B<cvscop> is displayed by the CVS command line program or
through the various logging facilities in a GUI CVS client (like WinCVS).

=head1 MODES

The I<mode> argument contains the name of the CVS administrative script that
launched B<cvscop>.  Some modes have special arguments that are automatically
appended by CVS, and B<cvscop> expects to see them when invoked in those
modes.

=over

=item commitinfo

Pre-commit phase, appropriate for user/filename/path policies.

=item verifymsg

Pre-commit phase for message content policies.

=item loginfo

Post-commit phase, useful for logging and external program launching.

=item taginfo

Only happens on cvs tag operations.

=item echo

Not an actual phase, this mode displays the data given to B<cvscop>.  Using
the --verbose option has a similar effect, except echo doesn't actually run
any checks

=back

=head1 OPTIONS

=over

=item -h, --help

Displays brief usage information.

=item --man

Attempts to extract and display this pod documentation as a man page.  Not the
recommended method of viewing this text, but it's much more convenient than
having to type C<pod2man cvscop.pl | nroff -man | less>, in case C<perldoc cvscop.pl> isn't working.

=item -v, --verbose

Causes B<cvscop> to emit a large amount of information regarding its input,
activites, regular expression tests, etc..  This option is useful for
debugging a policy entry in the configuration file, but displays what may be
too much information to a regular CVS user.

=item --sql

Enables SQL logging of CVS activites.  Applies to B<loginfo> mode only.  --sql
can be overridden by a <log> tag in the configuration file.

=item -r, --cvsroot I<path>

Specifies the local path to the CVS repository.  The I<path> option is
normally filled in with ${CVSROOT}.

=item -u, --user, --cvsuser I<name>

Identifies the CVS user name.

=item --module I<module>

To B<cvscop>, a module is actually a top-level directory in CVS.  B<cvscop>
will figure out what module is being used by the paths or filenames if they
are present, so the module argument is usually not necessary.

=item --svv I<${sVv}> (B<loginfo> only)

--svv provides the filenames and version numbers for each file being
processed.  It I<must> be the last argument in the administrative file, and
must be entered as B<--svv ${sVv}>

=back

=head1 CONFIGURATION FILE

The B<cvscop> configuration file (usually cvscop-config.xml) is XML containing
fail states or script triggers nested within pattern-match definitions.

The program reads the file as a tree, starting with the outermost tag
(the root) and working its way in to the innermost tags (the leaves).  If it
gets to a pettern match that fails, the tags down that branch will not be
followed.

See L<"EXAMPLES"> below for configuration examples.

=head2 <cvscop>[...]</cvscop>

The entire file must be contained within a <cvscop> tag.

=head2 <config>[...]</config>

Defines the configuration section of the file.  Configuration items are simple
string parameters, defined with <param> tags.

=head2 <param name=I<name> value=I<value> />

The <param> tag has two parameters: name and value.  All of the current
configurable options revolve around SQL logging:

=over

=item sql (on|off)

enables or disables SQL logging from loginfo mode.

=item dbhost

SQL server

=item dbport

MySQL port number

=item db

database name

=item dbuser

database login

=item dbpasswd
database password

=back

=head2 <ruleset>[...]</ruleset>

The <ruleset> tag encloses <match>, <deny>, and <run> tags.  A <ruleset> tag
can only be enclosed within the root <cvscop> tag.

=head2 <match>[patterns|subtags][...]</match>

<match> tags have two parameters: field and boolean.  They contain one or more
regular expressions and more <match>, <deny>, or <run> tags.  The tags
contained in a <match> tag will be skipped unless one or all (as configured by
the boolean parameter) of its conditions are set.

=head3 bool

Boolean match mode.  Can be "AND" or "OR".  "AND" states that all match
conditions must be met to continue, while "OR" requires only one condition to
be met.  "OR" is the default, and does not need to be specified.

=head3 field

The field parameter specifies which of the following are to be matched:

=over 4

=item action (B<taginfo> mode only)

Which tag operation: move, delete, add

=item mode (all modes)

Mode: B<commitinfo>, B<loginfo>, B<taginfo>, B<verifymsg>, B<echo>

=item cvsroot (all modes, requires --cvsroot option)

Local path to repository.  This is just a directory path, with no :pserver or
:ext, etc..  It is usually fed the argument ${CVSROOT}, which is translated by
CVS into the path.

=item cvsuser (all modes, requires --cvsuser option)

Login being used.  Usually take the argument ${USER}, which CVS translates
into the correct user name.

=item path (B<commitinfo>, B<taginfo>)

Path from the module to the file(s), exclusive (does not include module or
filename)

=item module (B<commitinfo>, B<taginfo>)

Top-level directory in the respository-- not a module as defined in the
modules file.

=item message (B<verifymsg>, B<loginfo>)

This string contains the full text of the log message.

=item tag (B<taginfo> only)

The name of the tag.

=item tempfile (B<verifymsg> only)

The path and filename of the temporary file containing the log message.  This
one isn't terribly useful.

=back

=head3 Regular expressions and the <match> tag

Regular expressions define a text pattern (Regular expressions are beyond the
scope of this document.  See L<perlre> for help).  In cvscop, simplified Perl
regular expressions are used.  The differences between cvscop and Perl regular
expressions are:

=over

=item

Using C<m/pattern/> is not allowed.  Use C</pattern/> instead.

=item

Prepending an exclamation mark applies a logical NOT to the
match.  For example, C<!/myname/> will match only if 'myname'
is not present.

=item

Only the C<i>, C<m>, C<s>, or C<x> modifiers can be appended to the pattern.

=item

Any whitespace will be collapsed down into single spaces.

=item

Escaping of some unusual characters will cause strange results.
This is a known issue, and it's being worked on.  For now,
running a test installation for some of these expressions may
help save some headaches.

=item

Some sequences like C<$1> and C<(?{CODE})> are disabled.

=item

Multiple expressions are defined in the same <match> by
space-delimiting them.

=item

If the parser determines that there are improperly-formatted
expressions, it will attempt (after complaining to STDERR) to
match the text content as a literal string (whitespace
notwithstanding). It is not recommended to use
non-regular-expression text in a <match> tag.

=back

=head2 <deny>[message]</deny>

The <deny> tag contains a failure message.  When its parent <match> tags have
passed, the text contained in the <deny> tag is displayed and cvscop exits
with a failure status, which causes the CVS operation to fail.  If no message
is provided, a generic "Permission Denied" message will be shown.

=head2 <run>[command] [arguments]</run>

When a <run> tag is reached, it causes an external program or script
(specified as the text inside the <run> tag) to be launched.
Optionally, cvscop can check the program or script's exit status
and exit with a fail status if appropriate.

=over

=item each=I<field>

If this attribute is set to any of the fields for <match>, it will fill '$_'
(see below) in with the appropriate field.  If it is set to B<ary_files>,
B<ary_version>, or B<ary_oldver>, the program or script will be run once each
for the file names, CVS revisions, or old revision numbers, respectively.  If
'$_' appears in the command arguments, it will be filled in accordingly.

=item errcode=I<(anything)>

If this attribute is present at all, the error code will be taken into account
after execution, and cvscop will exit upon failure.

=item <run> command line

The text content of a <run> tag contains the command and arguments to be run.

=item Variable replacement in <run> statements

The text of any <match> field can be inserted into a <run> tag's command
arguments by prepending the name of the field with a dollar-sign (e.g.
$cvsuser, $cvsroot).  The default variable ($_) is filled in by the
'each' argument (above).

=back

=head2 <!-- [comment] -->

Standard XML comments can be placed anywhere in the configuration file.

=head1 EXIT STATUS

Cvscop's exit status is used by CVS to determine whether an action will be
allowed.  Exit status will be 0 if no fail condition has been met, or 1 if a
<deny> tag has been reached.  Cvscop can also be made to inherit the exit
status from an external script or program launched by a <run> tag.


=head1 INSTALLATION

=over

=item 1. Check out a copy of the CVSROOT directory

=item 2. Place cvscop.pl in the CVSROOT working copy

The working copy of the cvscop.pl file must have the proper execute bit set to
cause the real file to be executable.

=item 3. Create cvscop-config.xml

See L<"CONFIGURATION FILE"> above and L<"EXAMPLES"> below.

=item 4. Edit commitinfo, loginfo, taginfo, and/or verifymsg

See L<"EXAMPLES"> below.  Putting cvscop control over the CVSROOT directory is
a bad idea, as a mistyped configuration file could cause you to be locked out
of your repository.  For this reason you should specifically have CVSROOT
launch /bin/true, and use DEFAULT instead of ALL for a global cvscop
configuration.

=item 5. Edit checkoutlist

Any new files created must be added to the checkoutlist.

=item 6. Commit

Remember to add the new files with C<cvs add>.  Upon commit all of the files listed in B<commitinfo> will be refreshed.

=back

=head1 EXAMPLES

Simple examples of commitinfo, loginfo, taginfo, and verifymsg with the
recommended options

  # commitinfo
  #
  CVSROOT /bin/true
  DEFAULT /path/to/CVSROOT/cvscop.pl commitinfo --cvsroot ${CVSROOT} --cvsuser ${USER}

  # loginfo
  #
  CVSROOT /bin/true
  DEFAULT /path/to/CVSROOT/cvscop.pl loginfo --cvsroot ${CVSROOT} --cvsuser ${USER} --svv %{sVv}

  # taginfo
  #
  CVSROOT /bin/true
  DEFAULT /path/to/CVSROOT/cvscop.pl taginfo --cvsroot ${CVSROOT} --cvsuser ${USER}

  # verifymsg
  #
  CVSROOT /bin/true
  DEFAULT /path/to/CVSROOT/cvscop.pl verifymsg --cvsroot=${CVSROOT} --cvsuser=${USER}


A simple cvscop-config.xml that fails everything (not recommended)

      <cvscop>
        <ruleset>
          <deny />
        </ruleset>
      </cvscop>

The same, with SQL logging enabled

      <cvscop>
        <config>
          <param name="sql" value="on" />
          <param name="db"       value="my_db_name" />
          <param name="dbuser"   value="my_db_login" />
          <param name="dbpasswd" value="my_db_password" />
          <param name="dbhost"   value="my_db_host" />
          <param name="dbport"   value="8888" />
        </config>
        <ruleset>
          <deny />
        </ruleset>
      </cvscop>

Limit commits to three users (larry, moe, and curly)

      <cvscop>
        <ruleset>
          <match field="mode">
            /^commitinfo$/
            <match field="cvsuser" bool="and">
              !/^larry$/
              !/^moe$/
              !/^curly$/
              <deny>
                Only larry, moe, or curly may commit!
              </deny>
            </match>
          </match>
        </ruleset>
      </cvscop>

...or for a more complex example, we have three engineers (Larry, Shemp, and
Curly) with private directories.  Commit access to these directories is
limited to the individual engineers and to their manager (Moe):

      <cvscop>
        <ruleset>
          <match field="mode">
            /^commitinfo$/
            
            <match field="path">
                /^private\/larry/
                <match field="cvsuser" bool="and">
                  !/^larry$/
                  !/^moe$/
                  <deny>
                    Commit to Larry's directory (private/larry) is denied!
                  </deny>
                </match>
            </match>
            
            <match field="path">
                /^private\/shemp/
                <match field="cvsuser" bool="and">
                  !/^shemp$/
                  !/^moe$/
                  <deny>
                    Commit to Shemp's directory (private/shemp) is denied!
                  </deny>
                </match>
            </match>
            
            <match field="path">
                /^private\/curly/
                <match field="cvsuser" bool="and">
                  !/^curly$/
                  !/^moe$/
                  <deny>
                    Commit to Curly's directory (private/curly) is denied!
                  </deny>
                </match>
            </match>
            
          </match>
        </ruleset>
      </cvscop>


Launch an external script for each file.  Note the $module, $path, and $_
variables in the command arguments.

      <cvscop>
        <ruleset>
          <match field="mode">
            /^loginfo$/
            <run each="ary_files">
              /home/curlyjoe/bin/myscript.sh $module/$path/$_
            </run>
          </match>
        </ruleset>
      </cvscop>



=head1 PREREQUISITES

XML::Parser

DBI

DBD::mysql

=head1 SEE ALSO

http://cvscop.sourceforge.net/

L<perlre>

L<cvs(1)>

=cut




#
#
# originally Tagtree.pm
#
# Creates a tree of references from XML input
#


# We start with Tags.  They contain the tag data and parent/child references.

package Tagtree_Tag;

sub new {
    my $class = shift;
    my $self  = {
                tagname    => '',       # name of the tag in question
                attributes => {},       # hash of text attributes
                text       => '',       # Non-XML content of a tag
                parent     => undef,    # reference to a parent tag
                children   => ()        # array of refs to children
                };
    bless $self, $class;
}



# Now the regular Tagtree package

package Tagtree;

use XML::Parser;

sub new {
    my $class = shift;
    my $self  = {
                nodes => ()    # Node refs go into an array
                };
    bless $self, $class;
}

# parse will read XML from a string and build a tree.

sub parse {
    my $self = shift;
    my $in   = shift;

    # invoke a parser

    my $parser = new XML::Parser( Style => 'Tree' );
    my $tree   = $parser->parse( $in );

    # walk the tree

    $self->drink_tree($tree->[0], $tree->[1], undef);
}

# drink_tree walks the whole tree and builds nodes

sub drink_tree {
    my $self      = shift;
    my $my_name   = shift;
    my $my_ary    = shift;
    my $my_parent = shift;
    my $newnode   = new Tagtree_Tag;

    push @{ $self->{nodes} }, $newnode;    # add us to the array

    my $my_index  = 1;  # where the children in the array start

    $newnode->{tagname}    = $my_name;
    $newnode->{parent}     = $my_parent;
    $newnode->{attributes} = $my_ary->[0];

    # deal with children and text

    while (defined $my_ary->[$my_index]) {

        if ($my_ary->[$my_index] eq '0') {          # text content
            my $my_text = $my_ary->[$my_index + 1];
            for ($my_text) {
                s/^\s+//sg; # Leading whitespace
                s/\s+$//sg; # Trailing whitespace
#                s/\n[ \t]+/\n/sg;   # indentation
                s/\s+/ /sg; # Grouped whitespace
#                s/ *\\n */\n/sg; # explicit '\n', no spaces
            }
            $newnode->{text} .= "\t" unless $newnode->{text} =~ /^$/;
            $newnode->{text} .= $my_text;
        } else {                                    # child tag
            push @{ $newnode->{children} }, $self->drink_tree(
                    $my_ary->[$my_index],
                    $my_ary->[$my_index + 1],
                    $newnode);                      # yow!
        }
        $my_index += 2;
    }

    return $newnode;
}




