// -*- Pike -*-
//
// YP (NIS) Functionality for Pike
// Per Hedbor & Henrik Grubbstrm 1996-1997
//

#pike __REAL_VERSION__
inherit .___Yp;

#if constant(.___Yp.Domain)

// Usual nicknames
constant nicknames = ([
  "passwd":"passwd.byname",
  "group":"group.byname",
  "networks":"networks.byaddr",
  "hosts":"hosts.byname",
  "protocols":"protocols.bynumber",
  "services":"services.byname",
  "aliases":"mail.aliases",
  "ethers":"ethers.byname"
]);

//! Network Information Service aka YP map.
class Map {
  object(Domain) domain;
  string ypmap;

  //! Create a new YP-map object.
  //!
  //! @[map] is the YP-map to bind to. This may be a nickname, such as
  //! @tt{passwd@} instead of just @tt{passwd.byname@}.
  //!
  //! If @[domain] is not specified, the default domain will be used.
  //!
  //! @note
  //! If there is no YP server available for the domain, this
  //! function call will block until there is one. If no server appears
  //! in about ten minutes or so, an error will be returned. The timeout
  //! is not configurable.
  //!
  void create(string map, string|void domain)
  {
    if (domain) {
      this::domain = Domain(domain);
    } else {
      this::domain = Domain();
    }
    ypmap = map;
    array err;
    do {
      // Force an error if the map does not exist.
      err = catch{
	this::domain->match(ypmap, "foo");
	return;
      };
    } while (ypmap = nicknames[ypmap]);
    throw(err);
  }

  //! Search for the key @[key]. If there is no @[key] in the map, 0 (zero)
  //! will be returned, otherwise the string matching the key will
  //! be returned.
  //!
  //! @note
  //! @[key] must match exactly, no pattern matching of any kind is done.
  //!
  string match(string key)
  {
    return(domain->match(ypmap, key));
  }
  string `[](string key)
  {
    return(domain->match(ypmap, key));
  }

  //! Returns the whole map as a mapping.
  mapping(string:string) all()
  {
    return(domain->all(ypmap));
  }

  mapping(string:string) cast_to_mapping()
  {
    return(domain->all(ypmap));
  }

  //! Call a function for each entry in the map.
  //!
  //! For each entry in the map, call the function @[fun].
  //!
  //! The function will be called like
  //!   @expr{void fun(string key, string value)@}.
  //!
  void map(function(string,string:void) fun)
  {
    domain->map(ypmap, fun);
  }

  //! Return the server that provides this map.
  string server()
  {
    return(domain->server(ypmap));
  }

  //! Return the order number for this map.
  int order()
  {
    return(domain->order(ypmap));
  }

  //! Return the number of entries in this map.
  int _sizeof()
  {
    return(sizeof(indices(domain->all(ypmap))));
  }

  //! Return the keys of the map.
  array(string) _indices()
  {
    return(indices(domain->all(ypmap)));
  }

  //! Return the values of the map.
  array(string) _values()
  {
    return(values(domain->all(ypmap)));
  }

  //! Convert the map to a mapping
  protected mapping cast(string type)
  {
    if( type=="mapping" )
      return cast_to_mapping();
    return UNDEFINED;
  }
}

#endif
