import Monotype;

// some modes we find useful
constant MODE_JUSTIFY = 0;
constant MODE_LEFT = 1;
constant MODE_RIGHT = 2;
constant MODE_CENTER = 3;

constant pneumatic_quad_code = "H J";

constant dicts = (["en_US": "hyph_en_US.dic",
		   "nl_NL": "hyph_nl_NL.dic",
		   "de_DE": "hyph_de_DE.dic",
		   "de_CH": "hyph_de_CH.dic",
       "fr_FR": "hyph_fr_FR.dic",
       "fr_FR": "hyph_es_ES.dic",
       "fr_FR": "hyph_it_IT.dic",
		]);

constant COLUMN_STRATEGY_EQUAL = 1; // prefer that all columns have an equal length.
constant COLUMN_STRATEGY_FILL = 0; // default, prefer to fill the first column before filling the second.

object hyphenator;
mapping hyphenation_rules = ([]);

Line current_line;

mapping config;

int interactive = 0;

int numline;
int pagenumber;
int linesonpage;

object JustifyingSpace;

mapping spaces = ([]);
mapping quadding_spaces = ([]);
mapping highspaces = ([]);
 
array(Line) _lines = ({});
array(Page) pages = ({});
array ligatures = ({});
mapping ligature_replacements_from = ([]);
mapping ligature_replacements_to = ([]);

string eheader_code = "";
string efooter_code = "";
string oheader_code = "";
string ofooter_code = "";

//! the matcase and stopbar objects
object m;
object s;

int modifier = 0;

int isitalics = 0;
int issmallcaps = 0;
int isbold = 0;
int canHyphenate = 1;

string d_code = "D";
string fine_code = "0005";
string coarse_code = "0075";

float space_adjust = 0.0;
int indent_adjust = 0;
int pad_units = 0; // units to add at beginning and end of line for safety in carrying
int allow_tight_lines = 0;

int line_mode = MODE_JUSTIFY;
float hanging_punctuation_width = 0.0;
string last = "";
array data_to_set = ({});

object space_regex;

mixed `->lines()
{
  return _lines;  
}

mixed `->lines=(mixed l)
{
  int height = config->height;
  if(height && l && sizeof(l) > height)
  {
    _lines = l[0..(height-1)];
  }
  else _lines = l;
  return l;
}

/*
  Settings (partial):
    setwidth
    linelengthp
    matcase
    stopbar
    mould
*/
void create(mapping settings)
{	

//  werror("Monotype.Generator(%O)\n", settings);
  float lineunits;
  
  if(settings->linelengthp && settings->lineunits)
  {
    throw(Error.Generic("Line length must be specified in picas or units, but not both.\n"));
  }
  else if(settings->linelengthp)
    lineunits = round(18 * (settings->pointsystem||12) * 
			(1/settings->setwidth) * settings->linelengthp);
  else if(settings->lineunits)
  {
    settings->linelengthp = lineunits/(18 * (settings->pointsystem||12) *
                         (1/settings->setwidth));
    lineunits = settings->lineunits;
  }    
  config = settings;
  config->lineunits = lineunits;

  if(settings->matcase)
    set_matcase(settings->matcase);
  if(settings->stopbar)
    set_stopbar(settings->stopbar);
  
//    werror ("line should be %O units.\n", lineunits);

  if(config->pad_margins)
  {
    pad_units = sort(indices(spaces))[-1];
    werror ("%d units will be used as padding at each end.\n", pad_units);
  }
  
  // set up the code substitutions for unit adding
  if(config->unit_adding)
  {
	  fine_code = "N K J";
	  coarse_code = "N K";
  }

  if(config->unit_shift)
  {
      d_code = "E F";
  }  
    
  load_hyphenator();
  
  if(config->hanging_punctuation)
  {
    if(catch(hanging_punctuation_width = sort(m->get_punctuation()->get_set_width())[0]))
    {
      werror("Unable to calculate hanging punctuation units.\n");
    }
    else
    {
      werror("Hanging punctuation will subtract %f units the length of text but not the overall line.\n", hanging_punctuation_width*2);
//      lineunits += (int)(hanging_punctuation_width * 2);
      config->hanging_punctuation_width = hanging_punctuation_width;
    }
  }
  
  if(!space_regex) 
    space_regex = Regexp.PCRE.Widestring("\\h");  
}

this_program clone(void|mapping overrides, void|int include_headfoot)
{
  mapping cfg = config - (<"linelengthp", "lineunits">);
  if(overrides) cfg = cfg + overrides;
  if(!(cfg->linelengthp && cfg->lineunits))
    cfg->lineunits = config->lineunits;
  object c = this_program(cfg);
  
  if(include_headfoot)
  {
    c->eheader_code = eheader_code;
    c->efooter_code = efooter_code;
    c->oheader_code = oheader_code;
    c->ofooter_code = ofooter_code;
  }
  
  return c;
}

void set_matcase(Monotype.MatCaseLayout mca)
{
  m = mca;
  
  load_ligatures(m);

  object js = m->elements["JS"];
	// houston, we have a problem!
  if(!js) error("No Justifying Space in MCA!\n");
  if(s)
    load_spaces(m);

  JustifyingSpace = RealJS(js);
}

void set_stopbar(Monotype.Stopbar stopbar)
{
  s = stopbar;
  if(m)
    load_spaces(m);
}

void set_hyphenation_rules(string rules)
{
  hyphenation_rules = ([]);
  foreach(rules/"\n";;string rule)
  {
    rule = String.trim_all_whites(rule);
    if(!strlen(rule))
      continue;
    rule = lower_case(rule);
    hyphenation_rules[rule - "-"] = rule;
  }
  
  werror("hyphenation rules: %O\n", hyphenation_rules);
}

protected void load_hyphenator()
{
  #if constant(Public.Tools.Language.Hyphenate.Hyphenate)
    string lang = "en";
    if(config->lang) lang = config->lang;
    if(config->hyphenate)
    {
//      werror("loading hyphenator " + dicts[lang] + "\n");
      hyphenator = Public.Tools.Language.Hyphenate.Hyphenate(combine_path(config->dict_dir, dicts[lang]));
    }
  #else
  werror("No hyphentation engine present, functionality will be disabled.\n");
  #endif
}

protected void load_spaces(object m)
{
  foreach(m->spaces;;object mat)
    spaces[s->get((mat->row_pos<16?mat->row_pos:12))] = mat;  
  
  if(spaces[9] && spaces[18])
     spaces[27] = 1;

  foreach(m->get_highspaces(s);;object matrix)
  {
    highspaces[matrix->set_width] = matrix;
  }
  
  if(config->unit_shift)
  {
    foreach(m->spaces;;object mat)
    {
      object ns = mat->clone();
      float new_width = (float)s->get(mat->row_pos-1 || mat->row_pos);
      ns->set_width = new_width;
      spaces[(int)new_width] = ns;
    }
  }
  
  if(config->maximum_quad_units)
  {
    foreach(indices(spaces);;int x)
      if(x <= config->maximum_quad_units)
        quadding_spaces[x] = spaces[x];
  }
  else 
  {
    quadding_spaces = spaces + ([]);
  }
  
//  werror("SPACES: %O\n", spaces);
}

protected void load_ligatures(object m)
{
    foreach(m->get_ligatures();; object lig)
    {
      ligatures += ({ ({lig->style||"R", lig->activator}) });	
    }

    foreach(ligatures;;array lig)
    {
//    	werror("lig:%O\n", lig);
      if(!ligature_replacements_to[lig[0]])
        ligature_replacements_to[lig[0]] = ({});
      if(!ligature_replacements_from[lig[0]])
        ligature_replacements_from[lig[0]] = ({});
      ligature_replacements_from[lig[0]] += ({ lig[1], "<A" + lig[1] + ">" });
      ligature_replacements_to[lig[0]] += (({ "<A" + lig[1] + ">" }) * 2 ); 
    }
  /*
    ligature_replacements_from = ligatures + map(ligatures, lambda(array a){return "<A" + a[1] + ">";});
    ligature_replacements_to = map(ligatures, lambda(string a){return "<A" + a + ">";}) * 2;
  */

//  werror("ligs from:%O\n", ligature_replacements_from);
//  werror("ligs to:%O\n", ligature_replacements_to);
}

protected array prepare_data(array data, void|StyledSort template)
{
  array out = allocate(sizeof(data));
  foreach(data; int i; mixed d)
  {
    if(d == " ")
      out[i] = JustifyingSpace;
    else if(space_regex->match(d))
      out[i] = JustifyingSpace;
    else
      out[i] = create_styled_sort(d, space_adjust, template);
  }
  
  return out;
}

//! @param input
//!  a native pike widestring (not utf8 encoded, etc)
void parse(string input)
{
  object parser = Parser.HTML();
  mapping extra = ([]);
  parser->_set_tag_callback(i_parse_tags);
  parser->_set_data_callback(i_parse_data);
  parser->set_extra(extra);

  // feed the data to the parser and have it do its thing.
  parser->finish(input);

  // put the footer at the end of the set text if we have one.
  if(config->page_length)
  {
	  insert_footer();
  }	
  update_page();    
}

// TODO: make this method re-entrant.
array lig_syl = ({});
mixed break_ligatures(string syl)
{
  lig_syl = ({});
  object parser = Parser.HTML();
  parser->_set_tag_callback(syl_parse_tags);
  parser->_set_data_callback(syl_parse_data);
  parser->finish(syl);
	
  return lig_syl;
}

mixed syl_parse_data(object parser, string data, mapping extra)
{
	lig_syl += data/"";
}

mixed syl_parse_tags(object parser, string data, mapping extra)
{
	if(Regexp.SimpleRegexp("<[A].*>")->match(data))
	{
		lig_syl += ({(data[2..sizeof(data)-2])});
	}
}

mixed i_parse_data(object parser, string data, mapping extra)
{
    // data_to_set is our setting buffer. when we arrive here, there may be things in it already,
    // such as ligatures and the like. we add whatever data this method was passed.

	// TODO: at the end of the document, we should verify that the setting buffer is empty.
	// possible situations where that might happen include ending the job with a ligature
	// where this callback wouldn't be called.

  if(in_column)
  {
    column_data += data;
    return 0;
  }
  else if(in_columnset)
  {
    // we don't want to keep any text inside of a column set but outside of a column.
    return 0;
  }
	if(in_header)
	{
		if(in_even)
		  eheader_code += data;
		if(in_odd)
		  oheader_code += data;
		return 0;
	}
	else if(in_footer)
	{
		if(in_even)
		  efooter_code += data;
		if(in_odd)
		  ofooter_code += data;
		return 0;
	}	

    string mod = "R";
    if(isitalics) mod = "I";
    else if (issmallcaps) mod = "S";
    else if (isbold) mod = "B";

    string xdata = replace(data, ({"\r", "\t"}), ({"\n", " "}));
    
    string dts = replace(xdata, ligature_replacements_from[mod] || ({}), ligature_replacements_to[mod] || ({}));

	if(dts !=  xdata) return dts;

    foreach(xdata/"\n";; data)
    {
  //    data = ((data / " ") - ({""})) * " ";
//werror("Ligatures: %O, %O", ligatures, map(ligatures, lambda(string a){return "<A" + a + ">";}));

      data_to_set += prepare_data(data/"");
    }

// note that we don't automatically process the buffer when we receive data, as there may be specifically added ligatures
// that won't appear as part of the current word but are logically part of it (which would affect hyphenation, in particular). 
// instead, we keep adding to the buffer until a code that wouldn't appear in the middle of a word arrives. that includes
// things like quads and line endings.
// 
// TODO: we currently also include changes in alphabet here, such as when we transition from roman to italics. that would prevent
// situations where, for example, we italicize part of a word from participating in hyphenation. does this occur often enough 
// to be of concern?
  //  process_setting_buffer();
}

void insert_header(int|void newpara)
{
	pagenumber++;
    linesonpage = 0;
	string header_code;
	if(pagenumber%2) header_code = oheader_code;
	else header_code = eheader_code;
	
	make_new_line();
    current_line->line_on_page = linesonpage;
	
	if(!in_do_header && sizeof(header_code))
	{
		in_do_header = 1;
		current_line->errors->append("* New Page Begins -");
	//	werror("parsing header: %O\n", header_code);
		array _data_to_set = data_to_set;
		data_to_set = ({});
		object parser = Parser.HTML();
		mapping extra = ([]);
		parser->_set_tag_callback(i_parse_tags);
		parser->_set_data_callback(i_parse_data);
		parser->set_extra(extra);

		// feed the data to the parser and have it do its thing.
		parser->finish(header_code);
		data_to_set = _data_to_set;
		in_do_header = 0;
	}
	make_new_line(newpara);
}

void insert_footer()
{
	string footer_code;
	if(pagenumber%2) footer_code = ofooter_code;
	else footer_code = efooter_code;
	
	make_new_line();
	
	if(!in_do_footer && sizeof(footer_code))
	{
		in_do_footer = 1;
	//	werror("parsing footer: %O\n", footer_code);
		array _data_to_set = data_to_set;
		data_to_set = ({});
		object parser = Parser.HTML();
		mapping extra = ([]);
		parser->_set_tag_callback(i_parse_tags);
		parser->_set_data_callback(i_parse_data);
		parser->set_extra(extra);

		// feed the data to the parser and have it do its thing.
		parser->finish(footer_code);
		data_to_set = _data_to_set;
		in_do_footer = 0;
	}
}

void reset()
{
  current_line = 0;
  lines = ({});
  pages = ({});  
  numline = 0;
  pagenumber = 0;
  linesonpage = 0;
  oheader_code = "";
  eheader_code = "";
  ofooter_code = "";
  efooter_code = "";
}

void add_column(Monotype.Line line)
{
	if(!current_line)
	{
	  // if we get here, and there's no line present, it's the first line of the job, thus, by default, a new paragraph.
	  insert_header(1);
	}
  current_line->add(line);
}

int process_setting_buffer(int|void exact)
{
	int lastjs = 0;
	object tightline;
//	object phline;
	int tightpos;
	
	if(!current_line)
	{
	  // if we get here, and there's no line present, it's the first line of the job, thus, by default, a new paragraph.
	  insert_header(1);
	}
// werror("data to set: %O\n", data_to_set);
  for(int i = 0; i<sizeof(data_to_set) ;i++)
	{
//  	int additback;
//  	object hpspace;

	  tightline = 0;
	  tightpos = 0;
//	  werror("data_to_set[%d]: %O\n", i, data_to_set[i]);
	  if(data_to_set[i]->is_real_js)
    {
      lastjs = i;
      // we don't want to allow duplicated justifying spaces, nor justifying spaces first on the line.
	    if(current_line->elements && (sizeof(current_line->elements) && current_line->elements[-1]->is_real_js) || !current_line->non_spaces || !sizeof(current_line->elements)) 
		  {
	      continue;
		  }
    }
    else
    {
      // we do this last, so that it's always the last item on the line and thus first to be removed
    	// if the first real sort on the line is a punctuation mark.
      if(hanging_punctuation_width != 0.0)
      {
        float res;
       // werror("character: %O\n", data_to_set[i]->character);
        
        // warning! bad form follows in favor of clearer logic.
        if(!(sizeof(current_line->elements) || current_line->non_spaces)) 
        {
          if(!m->is_punctuation(data_to_set[i]->character))
          {
            // we add the front and back widths up front, and then move them around later, if necessary.
            res = low_quad_out(hanging_punctuation_width);
            if(res != hanging_punctuation_width)
            {
              throw("unable to add " + hanging_punctuation_width + " units of space to hanging punctuation.\n");
            }
          }          
        }
      }
            
      current_line->non_spaces++;
    } 
    
//    werror("adding %O\n", data_to_set[i]);
	  current_line->add(data_to_set[i]);

    // if permitted, prepare a tight line for possible use later.
    
    object removed_sort;
    
    
    // if we've overset a line, and the thing that threw us over was a justifying space; it can be dropped 
    // and a new line started because the previous sort fit.
	  if(current_line->is_overset())
	  {
	    removed_sort = current_line->remove();
	    
	    if(removed_sort->is_real_js)
 	    {
  	    werror("skipping to a new line.\n");
  	    new_line();
  	    continue;
	    }
      else current_line->add(removed_sort);
	  }
	  
	  if(current_line->is_overset() && config->enable_combined_space)
	  { 
      // first, let's see if removing 1 unit from each justifying space will work.
     	object tl = Line(m, s, config + (["combined_space": 1, "lineunits": pad_units?(config->lineunits-(pad_units*2)):config->lineunits]), this);
	   	tl->re_set_line(current_line);
//	   	werror("current_line: %O\n", current_line->render_line(1));
	     	
      int j = i;

      if(lastjs != i) // we're not at the end of a word, here, folks.
      {
 	      while(sizeof(data_to_set)>++j && data_to_set[j] != JustifyingSpace)
 	      {
//		werror("adding to tight line: %O\n", data_to_set[j]);
          tl->add(data_to_set[j]);
        } 
      }
 	     
	if(!tl->is_overset()) // did the word fit using combined spaces?
    	{
    	  werror("tight line fit2!\n");
     	  tl->line_number = current_line->line_number;
     	  tl->line_on_page = current_line->line_on_page;
    	  tightline = tl;	 
    	  tightpos = j;
    	  if(allow_tight_lines)
    	  {
    	    current_line = tl;
    	    i = tightpos;
		      new_line();
		      continue;
    	  } 
    	   
	     }
	     else
	     {
	       werror("tight line didn't fit.\n");
	     }
    }
     
	  if(current_line->is_overset()) // back up to before the last space.
	  {
	    werror("word didn't fit, justification is %d/%d\n", current_line->big, current_line->little);
      object x;

		  int can_try_hyphenation = 0;
		  if(!current_line->hyphenation_disabled())
		  {
  		  if(numline && sizeof(lines) && (!lines[-1]->is_broken || !current_line->can_justify()))
	  	    can_try_hyphenation = 1;
		    else if(config->unnatural_word_breaks)
		      can_try_hyphenation = 1; 
      }

      do
		  {
			  x = current_line->remove();
		  }
      while(x && x!= JustifyingSpace);

	    werror("removed word, justification is %d/%d\n", current_line->big, current_line->little);
		  if(exact) return 1;

      // prepare a clone of the line, so that we can jump back easily.
//	    phline = Line(m, s, config, this);
//	   	phline->re_set_line(current_line);
     	
     	
		  if(line_mode)
		  {
		    quad_out();
		    can_try_hyphenation = 0;
		  }

		  i = lastjs||-1; 
		  // if we backed up to the beginning of the setting buffer, that is, there isn't
			// a justifying space in it, we need to back up one more so that we're starting
			// back at the beginning of the buffer, rather than at the js, which we'd skip
			// once we start the next iteration of this loop.

		  // if we can't justify, having removed the last word, see if hyphenating will help, regardless if we hyphenated the last line.		
      
		  if(can_try_hyphenation)
			{	
			  int prehyphenated;
      	
			  int bs = search(data_to_set, JustifyingSpace, i+1);
			  if(bs!=-1)
			  {
			    array wordsorts = data_to_set[i+1..(bs-1)];
				  string word = (wordsorts->character)*"";
				  werror("attempting to hyphenate word %O from %O to %O\n", word, i, bs);

				  array word_parts;
				  if(search(word, "-") != -1)
				  {
				    word_parts = word / "-";
				    prehyphenated = 1;
				  }
				  else
				  {
  				  word_parts = hyphenate_word(word);				    
				  }
				  werror("word parts are %O\n", word_parts * ", ");

				  // if there are no hyphenation options, we're stuck.
				  if(sizeof(word_parts)<=1)
				  {
			      if(!current_line->can_justify() && tightline) // if we can't fit the line by breaking, see if there's a tightly justified line that will work.
  			    {
  			      werror("can't justify with regular word spaces, but we can with combined spaces, so let's do that.\n");
  			      current_line = tightline;
  			      i = tightpos;
  			      new_line();
  			      continue;
  			    }			  
				  }

				  else // there are syllables in this word.
				  {
					  array new_data_to_set = data_to_set;
					  int new_i = i;
					  int new_lastjs = lastjs;
					  int final_portion; // the last segment of the word being hyphenated
            
					  // for each word part, attempt to add the word, starting with the maximum parts.
					  for(final_portion = sizeof(word_parts)-2; final_portion >=0; final_portion--)
					  {
					    // first, we need to get the sorts that make up this syllable. it's possible that the syllable
					    // contains ligatures, so we might not be able to fully 
					    string portion = (word_parts[0..final_portion] * (prehyphenated?"-":""));
					    string carry = "";
					    int currentsort = 0;
					    array sortsinsyllable = ({});
			  		  int simple_break;
				  	  object brokenlig;
					    string syl;
					  
					    werror("portion: %O\n", portion);
					  
  					  foreach(portion/""; int x; string act)
	  				  {
		  			    carry += word[x..x];
//		  			    werror("carry: %O, sort: %O\n", carry, wordsorts[currentsort]->character);
			  		    if(wordsorts[currentsort]->character == carry)
				  	    {
					        sortsinsyllable += ({wordsorts[currentsort]});
					        carry = "";
  					      currentsort++;
					      }
  					  }
					  
					    werror("sortsinsyllable: %O\n", sortsinsyllable);
					    werror("sortsinsyllable: %O\n", sortsinsyllable->character);

					    if(lower_case(sortsinsyllable->character * "") == portion)
					    {
					      // we have the whole shebang. no need to mess with re-ligaturing.
					      werror("simple break.\n");
					      data_to_set = ({JustifyingSpace});
					      data_to_set += sortsinsyllable;
					      simple_break = 1;
					    }
					    else
					    {
					      werror("sorts in syllable = %O, portion = %O\n", sortsinsyllable->character, portion);
					      // we don't have the whole portion. most likely, the end of the portion breaks a ligature.
					      // so, we look at the sort in wordsorts after the last in the sortsinsyllable.
					      data_to_set = ({JustifyingSpace});
					      data_to_set += sortsinsyllable;

  					    brokenlig = wordsorts[sizeof(sortsinsyllable)];
	  				  
	  				    werror("broken ligature: %O\n", brokenlig->character);
	  				    
	  				    // syl is the part of the syllable containing a hanging ligature.
  					    syl = (portion[sizeof(sortsinsyllable->character * "")..]);
  					    //werror("syl: %O\n", syl);
  					    string modifier = "R";
  					    
  					    if(sizeof(sortsinsyllable))
  					       modifier = sortsinsyllable[-1]->get_modifier();
  					       
		  				  string lsyl = replace(syl, ligature_replacements_from[modifier]||({}), ligature_replacements_to[modifier]||({}));
 
	  					  if(syl != lsyl)
  						  {
		  					  // we have a ligature in this word part. it must be applied.
			  				  data_to_set += prepare_data(break_ligatures(lsyl), brokenlig);
				  		  }
					  	  else data_to_set +=  prepare_data(syl/"", brokenlig);
					    }

              // add a hyphen in the style of the last sort added.
	  					if(!(config->unnatural_word_breaks && config->hyphenate_no_hyphen))
	  					{
	  					  object template;
	  					  int i = -1;
	  					  do
	  					  {
	  					    template = current_line->elements[i];
	  					    werror(string_to_utf8(sprintf("template: %O\n", template)));
	  					    i--;
	  					  } while(template->is_real_js);
	  					  
		  				  data_to_set += prepare_data(({"-"}), template);
	  				  }
			 	    
			  		  werror("seeing if %O will fit... %O = %O", portion, data_to_set, data_to_set->character);
				  	  int res = process_setting_buffer(1);
					    if(!res)
					    {
						    werror("yes!\n");
						    // it fit! now we must put the rest of the word in the setting buffer.
						    if(sizeof(word_parts)>=final_portion)
						    {	
						    //  werror("simple_break: %O, prehyphenated: %O, wordsorts: %O, sortsinsyllable: %O, final_portion: %O\n", simple_break, prehyphenated, wordsorts, sortsinsyllable, final_portion);
						      if(simple_break)
						      {
    						    if(prehyphenated)
      						    data_to_set += wordsorts[sizeof(sortsinsyllable) + 1 /*hyphen*/ ..];
                    else
  						        data_to_set = wordsorts[sizeof(sortsinsyllable)..];
						      }
						      else
						      {
						        // it gets more complex here, as we have to put the second half of the broken ligature in first.						          
						        data_to_set = prepare_data((brokenlig->character[sizeof(syl)..])/"", brokenlig) + wordsorts[sizeof(sortsinsyllable)+1 ..];
						      
						        // TODO
						        // there is a chance that we may have to rescan for ligatures if the trailing portion of the broken ligature
						        // combines with the first sort(s) of the next word part to form a ligature itself.
						      }						    
						    }
						    
						    data_to_set += prepare_data(({" "}));
  					    data_to_set += new_data_to_set[bs+1..];
	
					      i = -1;
						    current_line->is_broken = 1;
						    break;
					    }
					
					    else // take it all off the line and try again.
					    {
						    werror("nope.\n");
					    }

  					  if(final_portion == 0 && !current_line->can_justify()) // we got to the last syllable and it won't fit. we must have a crazy syllable!
	  					{
		  				  if(tightline) // if we can't fit the line by breaking, see if there's a tightly justified line that will work.
			  			  {
				  		    werror("can't justify with regular word spaces, but we can with combined spaces, so let's do that.\n");
					  	    current_line = tightline;
						      i = tightpos;
						      new_line();
						      continue;
						    }
                else
  						    error(sprintf("unable to fit syllable %O on line. unable to justify.\n", word_parts[0]));
					    }
					    else if(final_portion == 0)
					    {
					      werror("sadness.\n");
					      array toadd = ({});
					      if(sizeof(new_data_to_set) > (bs+1))
					        toadd = new_data_to_set[bs+1..];
//werror("toadd: %O\n", toadd);
							  string lsyl = replace(word, ligature_replacements_from[wordsorts[-1]->get_modifier()]||({}), ligature_replacements_to[wordsorts[-1]->get_modifier()]||({}));
//werror("word: %O, lsyl: %O\n", word, lsyl);
//werror("wordsorts: %O\n", wordsorts);
							  if(word != lsyl)
							  {
								  // we have a ligature in this word part. it must be applied.
								  data_to_set = prepare_data((({" "}) + break_ligatures(lsyl)) + ({" "}), wordsorts[-1]) + toadd;
							  }
  							else data_to_set = prepare_data(({" "}), wordsorts[-1]) + wordsorts + prepare_data(({" "}), wordsorts[-1])  + toadd;
//werror("data_to_set: %O\n", data_to_set);
  						  i = -1;
	  				  }
            }						
			  	}
			  }
		  } 
      if(!sizeof(current_line->elements))
      {
        throw(Error.Generic(sprintf("Cannot fit anything on line of length %O units.\n", config->lineunits)));
      }
		  new_line();
	  }
  }
	
	data_to_set = ({});
	return 0;
}

int in_do_footer;
int in_do_header;
int in_footer;
int in_header;
int in_even;
int in_odd;
int in_column;
int in_columnset;
string column_data = "";
object column_parser;
array column_set = ({});
int cs_gutter;
int cs_height;

// TODO: this is just aweful. we need to come up with something a little more robust.
mixed i_parse_tags(object parser, string data, mapping extra)
{
    string lcdata = lower_case(data);

      if(in_column && !in_columnset && lcdata == "</columns>")
      {
        werror("GOT END OF COLUMNS\n");
        in_column--;
        if(in_column == 0)
        {
          int cols;
          werror("ACTUAL END OF COLUMNS.\n");
          
          // if the column strategy is "level", we have to calculate the length 
          // of the last page iteratively, unless we limit the columns to be of equal 
          // length. since that seems arbitrary, we offer this algorithm:
          //
          // keep shortening the page length until the last column is 
          // longer than the first column, then lengthen the page lenth by 1.
          if(column_parser->config->column_strategy == COLUMN_STRATEGY_EQUAL)
          {
            do
            {
              column_parser->parse(column_data);
              werror("pages: %O\n", column_parser->pages);
              object last_page = column_parser->pages[-1];
              array cnt = allocate(sizeof(column_parser->config->widths));
              foreach(last_page->lines;; object l)
              {
                cnt[l->col_number]++;
              }              
              werror("col alignment: %O\n", cnt);
              if(abs(cnt[-1] - cnt[0]) < sizeof(cnt)) // ok, as close as it's gonna get!
                break;
//                sleep(5);
              if(sizeof(column_parser->config->page_length) != sizeof(column_parser->pages))
              {
                int fpl = column_parser->config->page_length[0];
                column_parser->config->page_length = allocate(sizeof(column_parser->pages), column_parser->config->page_length[-1]);
                column_parser->config->page_length[0] = fpl;
              }
              column_parser->config->page_length[-1]--;
              column_parser->reset();
            } while(1);
            
          }
          else
          {
            column_parser->parse(column_data);
          }
          werror("columns contain %O lines.\n", sizeof(column_parser->lines));
          cols = sizeof(column_parser->config->widths);
          int page = 0;
          
          do
          {
            
            int ps = page?1:0;
            if(sizeof(column_parser->config->page_length) > page) ps = page;
            
          for(int l = 0; l < column_parser->config->page_length[ps]; l++)
          {
            if(sizeof(column_parser->lines) < l) break;
            
             for(int c = 0; c < cols; c++)
             {
               if(sizeof(column_parser->lines)<= ((column_parser->config->page_length[ps] * c) + l))
               {
                 low_quad_out(column_parser->config->widths[c]);
               }
               else
               {
                 add_column(column_parser->lines[l + (column_parser->config->page_length[ps] * c)]);
               }
                 if((c+1) < cols)
                 {
                   float g = column_parser->config->gutter;
                   if(column_parser->config->gutter > current_line->min_space_units)
                   {
                     current_line->add(JustifyingSpace);
                     g -= current_line->min_space_units;
                   }
                   low_quad_out(g);
                 }
                 else 
                 {
                   //quad_out();
                   new_paragraph();
                 }
             }  
          }
          
          if(sizeof(column_parser->lines) > (column_parser->config->page_length[ps] * cols))
            column_parser->lines = column_parser->lines[(column_parser->config->page_length[ps] * cols)..];
          else column_parser->lines = ({});

          page++;
        } while(sizeof(column_parser->lines));
          
        return 0;
      }
    }

    else if(in_columnset && lcdata == "</columnset>")
    {
      werror("GOT END OF COLUMNSET\n");
      in_columnset = 0;
      
      int max_lines;
      foreach(column_set;; object c)
      {
        int l = sizeof(c->lines);
        if(l > max_lines) max_lines = l;
      }
      
      for(int i = 0; i < max_lines; i++)
      {
        foreach(column_set; int cn; object c)
        {
          if(sizeof(c->lines) > i)
          {
            if(cn != 0)
              low_quad_out((float)cs_gutter);
            add_column(c->lines[i]);
          }
          else
          {
            low_quad_out((float)c->config->widths[0]);
          }
        }
        quad_out();
        new_paragraph();
      }
      column_set = ({});
      return 0;
    }
    else if(in_columnset)
    {
       if(has_prefix(lcdata, "<column "))
       {
         in_column = 1;
         mapping atts = parse_tag(data);
         if(!(int)atts->width)
           throw(Error.Generic("No column width specified.\n"));
           
         int width = (int)atts->width;   
         if(!width)
          throw(Error.Generic("Invalid column width specified.\n"));
          
         object cp = clone((["widths": ({width}), "height": cs_height, "gutter": cs_gutter, "column_strategy": COLUMN_STRATEGY_FILL, "pad_margins": 0, "page_length": ({config->page_length, config->page_length}) ]));
      	 column_data = "";
         column_set += ({cp});
         return 0;
       } 
       else if(lcdata == "</column>")
       {
         in_column = 0;
         column_set[-1]->parse(column_data);
       }
    
       else if(in_column)
       {
         column_data += data;
       }
       
       return 0;
    }
    
    else if(in_column)
    {
      column_data += data;
      return 0;
    }
    
  switch(lcdata)
  {
    case "<footer>":
		  in_even = 1;
		  in_odd = 1;
		  in_footer = 1;
		  return 0;
    
    case "</footer>":
      in_footer = 0;
		  return 0;

    case "<header>":
      in_even = 1;
      in_odd = 1;
      in_header = 1;
      return 0;

    case "</header>":
		  in_header = 0;
		  return 0;
      
    case "<ofooter>":
      in_even = 0;
      in_odd = 1;
      in_footer = 1;
	    return 0;

    case "</ofooter>":
		  in_footer = 0;
		  return 0;
    
    case "<oheader>":
		  in_even = 0;
		  in_odd = 1;
		  in_header = 1;
		  return 0;

    case "</oheader>":
      in_header = 0;
      return 0;

    case "<efooter>":
		  in_even = 1;
      in_odd = 0;
      in_footer = 1;
      return 0;

    case "</efooter>":
		  in_footer = 0;
		  return 0;

    case "<eheader>":
		  in_even = 1;
		  in_odd = 0;
		  in_header = 1;
		  return 0;

    case "</eheader>":
      in_header = 0;
      return 0;
  }

    if(in_footer)
    {
		if(in_even)
		  efooter_code += data;
		if(in_odd)
		  ofooter_code += data;
		return 0;
    }

    if(in_header)
    {
		if(in_even)
  		  eheader_code += data;
		if(in_odd)
		  oheader_code += data;
		return 0;
    }
	
	if(lcdata == "<i>")
	{
		isitalics ++;
	}
	else if(lcdata == "</i>")
	{
		isitalics --;
		if(isitalics < 0) isitalics = 0;
	}
	else if(lcdata == "<b>")
	{
		isbold ++;
	}
	else if(lcdata == "</b>")
	{
		isbold --;
		if(isbold < 0) isbold = 0;
	}
   else if(lcdata == "<sc>")
    {
       issmallcaps ++;
    }
   else if(lcdata == "</sc>")
    {
      issmallcaps --;
      if(issmallcaps < 0) issmallcaps = 0;
    }

   else if(lcdata == "<allowtightlines>")
    {
      werror("ENABLING TIGHT LINES\n");
      allow_tight_lines = 1;  
    }

   else if(lcdata == "</allowtightlines>")
    {
      werror("DISABLING TIGHT LINES\n");
      allow_tight_lines = 0;  
    }

  else if(lcdata == "<nohyphenation>")
  {
    werror("disABLING HYPHENATION\n");
    canHyphenate = 0;  
  }

  else if(lcdata == "</nohyphenation>")
  {
    canHyphenate = 1;  
  }
  
	else if(lcdata == "<left>")
	{
		process_setting_buffer();
		line_mode = MODE_LEFT;
	}
	else if(lcdata == "</left>")
	{
		process_setting_buffer();
		line_mode = MODE_JUSTIFY;
	}
	else if(lcdata == "<center>")
	{
		process_setting_buffer();
		line_mode = MODE_CENTER;
	}
	else if(lcdata == "</center>")
	{
		process_setting_buffer();
		line_mode = MODE_JUSTIFY;
	}
	else if(lcdata == "<right>")
	{
		process_setting_buffer();
		line_mode = MODE_RIGHT;
	}
	else if(lcdata == "</right>")
	{
		process_setting_buffer();
		line_mode = MODE_JUSTIFY;
	}
	else if(lcdata == "<justify>")
	{
		process_setting_buffer();
		line_mode = MODE_JUSTIFY;
	}
	else if(lcdata == "</justify>")
	{
		process_setting_buffer();
		line_mode = MODE_LEFT;
	}
	else if(lcdata == "<qo>")
	{
	  process_setting_buffer();
		
		new_paragraph(1);
    }
	else if(lcdata == "<p>")
	{
	  process_setting_buffer();	
	  new_paragraph();
    }
	// insert fixed spaces
  else if(Regexp.SimpleRegexp("<[sS][0-9]*>")->match(data))
	{
		process_setting_buffer();
		int toadd = (int)(data[2..sizeof(data)-2]);
    float added = low_quad_out((float)toadd);
		if((float)added != (float)toadd)
		{
			current_line->errors->append(sprintf("Fixed space (want %f units, got %f) won't fit on line... dropping.\n", (float)toadd, added));
		}
	}
	else if(Regexp.SimpleRegexp("<[sS][tT][0-9]*>")->match(data))
	{
		process_setting_buffer();
		float toset = (float)(data[3..sizeof(data)-2]);
		werror("requesting space to %O\n", toset);
		if(toset > current_line->lineunits)
		{
			current_line->errors->append(sprintf("Cannot add space beyond end of line. Requested %f, trimming to %O\n", (float)toset, current_line->lineunits));		  
      toset = current_line->lineunits;
		}
		if(current_line->linelength > toset)
		{
			current_line->errors->append(sprintf("Line set beyond requested units, skipping: requested %f, line contains %O\n", (float)toset, current_line->linelength));		  
      toset = 0;
		}
	  
	  // TODO: we need to get the justifying spaces adjusted appropriately (either remove the S code and make them fixed 
	  // spaces, or calculate the justification constant required to make them the width of the "placeholder".)
	  float toadd = toset - current_line->linelength;
	  if(toadd < 0.0) toadd = 0.0;
	  current_line->set_fixed_js(1);
	  werror("spacing to %O/%f/%f/%O\n", toset, toadd, current_line->linelength, current_line->lineunits);
    float added = low_quad_out((float)toadd);
	  werror("/spacing to %O/%f/%f/%O\n", toset, toadd, current_line->linelength, current_line->lineunits);
		if((float)toadd - (float)added >= 1.0)
		{
			current_line->errors->append(sprintf("Fixed space (want %f units, got %f) won't fit on line... dropping.\n", (float)toadd, added));
		}
	}
  
	// indent
	else if(Regexp.SimpleRegexp("<indent[\\-0-9]*>")->match(lcdata))
	{
	  indent_adjust = (int)(data[7..sizeof(data)-2]);
		process_setting_buffer();
	}
	// end letterspacing
	else if(Regexp.SimpleRegexp("</indent[\\-0-9]*>")->match(lcdata))
	{
		indent_adjust = 0;
		process_setting_buffer();
	}

	// letterspacing
	else if(Regexp.SimpleRegexp("<[Ll].*>")->match(data))
	{
//		process_setting_buffer();
		string sa = (data[2..sizeof(data)-2]);
		int w, f;
		sscanf(sa, "%d.%d", w, f);
		werror("sa: %s, w: %d, f: %d\n", sa, w, f);
		if(!(<0, 5>)[f])
		  throw(Error.Generic("Invalid adjustment " + sa + ". Only whole and half unit adjustments allowed.\n"));
		sscanf(sa, "%f", space_adjust);	
	}
	// end letterspacing
	else if(Regexp.SimpleRegexp("</[Ll].*>")->match(data))
	{
//		process_setting_buffer();
		space_adjust = 0.0;
	}
	// insert an activator
	else if(Regexp.SimpleRegexp("<[Aa].*>")->match(data))
	{
    if(data[2..sizeof(data)-2] == "JS")
      data_to_set += ({ JustifyingSpace });
    else
      data_to_set+= ({(create_styled_sort(data[2..sizeof(data)-2], space_adjust))});
	}
	else if(has_prefix(lcdata, "<columnset"))
	{
	  mapping atts = parse_tag(data);
    if(!atts->gutter)
      throw(Error.Generic("No gutter specified.\n"));
      
    int minspace = sort(indices(spaces))[0];
    int g = (int)atts->gutter;
      
    if(g != 0 && g < minspace)
      throw(Error.Generic("columnset gutter " + g + " is too small.\n"));

    if(has_index(atts, "height") && (int)atts->height < 1)
      throw(Error.Generic("columnset height must be a positive number.\n"));
    else
      cs_height = (int)atts->height;    

    cs_gutter = g;
    in_columnset = 1;
    return 0;  
	}
	else if(has_prefix(lcdata, "<columns"))
	{
	  array(int) widths;
	  int gutter;
    int count;
    int strategy;
    
    int minspace = sort(indices(spaces))[0];
    
	  mapping atts = parse_tag(data);
	  if(atts->strategy)
	  {
	    switch(lower_case(atts->strategy))
	    {
	      case "equal":
	        strategy = COLUMN_STRATEGY_EQUAL;
	        break;
  	    case "fill":
  	      strategy = COLUMN_STRATEGY_FILL;
  	      break;
  	    default:
  	      throw(Error.Generic("Unknown column strategy '" + lower_case(atts->strategy) + "'.\n"));
	    }  
	  }
	  
	  if(atts->count)
	  {
	    count = (int)atts->count;
	   
	    if(atts->width)
	    {
	      int w;
	      if((count * w + (count-1) * minspace) > config->lineunits)
	        throw(Error.Generic(count + " column set of width " + atts->width + " too wide for line.\n"));
	        
	      widths = ({ w }) * count;
	      gutter = (config->lineunits - (w*count)) / (count-1);
	    }
	    else if(atts->gutter)
	    {
	      int g = (int)atts->gutter;
	      
	      if(g != 0 && g < minspace)
	        throw(Error.Generic("gutter " + gutter + " is too small.\n"));
	        
	      if(((count * minspace) + (count-1) * gutter) > config->lineunits)
	        throw(Error.Generic(count + " column set with gutter " + atts->gutter + " too wide for line.\n"));
	      
	      gutter = g;
	      widths = ({((config->lineunits - (gutter * (count-1))) / count)}) * count;
	    }
	    else
	    {
	      throw(Error.Generic("columns using count must specify either width or gutter.\n")); 
	    }	      
	  }
	  else if(atts->widths)
	  {
	    int tot;
	    array wx = atts->widths / ",";
	    widths = allocate(sizeof(wx));
	    foreach(wx;int i; string w)
	    {
	      w = String.trim_all_whites(w);
	      if(!(int)w)
	      {
	        throw(Error.Generic("column " + (i+1) + " width " + w + " is not a whole number.\n"));
	      }
	      widths[i] = (int)w;
	      tot+=widths[i];
	    }
	    if(tot > config->lineunits && (tot + ((count-1) * minspace)) > config->lineunits)
	    {
	      throw(Error.Generic("total column widths are too wide for line.\n"));
	    }
	    
	    count = sizeof(widths);
	    gutter = (config->lineunits - tot) / (count-1);
	  }
	  
	  werror("columns lineunits = %O, count = %d, width = %O, gutter = %O", config->lineunits, count,  widths, gutter);
	  werror("STARTING COLUMNS\n");
	  // the page length array is the number of lines left on the current page to spread columns across, followed by full length pages.

	  column_parser = clone(0, 1);

	  column_parser->break_page();

    werror("headers and footers consume %d lines.\n", sizeof(column_parser->lines));

	  column_parser = clone((["widths":widths, "gutter": gutter, "column_strategy": strategy, "pad_margins": 0, "page_length": ({config->page_length-(linesonpage + sizeof(column_parser->lines)), config->page_length - sizeof(column_parser->lines)}) ]));
	  column_data = "";
	  in_column++;
	  return 0;
	  
	}
	else if(has_prefix(lcdata, "<setpagenumber "))
	{
		int matches, pn;
		matches = sscanf(lcdata, "<setpagenumber %d%*s>", pn);
		if(matches)
		  pagenumber = (pn-1); // we always increment before going into header, so account for that here.
		else
			current_line->errors->append("Failed to set page number, unable to extract desired number.\n");
			
	}
	else if(lcdata == "<pagenumber>")
	{
	   		  data_to_set += prepare_data(((string)pagenumber)/"");
	}
	else if(lcdata == "<romanpagenumber>")
	{
	   		  data_to_set += prepare_data((String.int2roman(pagenumber))/"");
	}
	else if(lcdata == "<lowercaseromanpagenumber>")
	{
	   		  data_to_set += prepare_data(lower_case(String.int2roman(pagenumber))/"");
	}
    else if(lcdata == "<pagebreak>")
	{
	   		 break_page();
	}	
}


//! a generic function that will extract attributes from an html-like tag.
mapping parse_tag(string tag)
{
  mapping data = ([]);

  sscanf(tag, "<%*s%*[ ]%s", tag);
  if(sizeof(tag))
    tag = tag[0..<1];
    werror("tag: %O\n", tag);
  string key = "", value = 0, delimiter = 0;
  if(sizeof(tag))
  {
    int inkey, inval, beforekey = 1, beforeeq, beforeval;
    foreach(tag/"";; string c)
    {
      if(beforekey)
      {
        if(space_regex->match(c)); // do nothing
        else
        {
          beforekey = 0;
          inkey = 1;
          key += c;
        }
      }
      else if(inkey)
      {
        if(space_regex->match(c))
        {
          inkey = 0;
          beforeeq = 1;
        }
        else if(c == "=")
        {
          inkey = 0;
          beforeval = 1;
        }
        else key += c;
      }
      else if(beforeeq)
      {
        if(space_regex->match(c)); // do nothing
        else if(c == "=")
        { 
          beforeeq = 0;
          beforeval = 1;
        }
        else
        {
          beforeeq = 0;
          inkey = 1;
          data[key] = "1";
          key = c;
        }
      }
      else if(beforeval)
      {
        if(space_regex->match(c)); // do nothing
        else if(c == "\'" || c == "\"")
        {
          delimiter = c;
          beforeval = 0;
          inval = 1;
          value = "";
        }
        else
        {
          delimiter = 0;
          inval = 1;
          beforeval = 0;
          value = c;
        }
      }
      else if(inval)
      {
        // no escaping of delimiters, yet.
        if((!delimiter && space_regex->match(c)) || (delimiter && c == delimiter))
        {
          inval = 0;
          beforekey = 1;
          data[key] = value;
          delimiter = 0;
          key = "";
          value = 0;
        }
        else 
        {
          value += c;
        }
      }
    }

  }
  
  if(sizeof(key)) data[key] = value?value:"1";
  return data;
}

array hyphenate_word(string word)
{
  // defer to custom hyphenations, first.
  if(hyphenation_rules)
  {
    object regex = Regexp.PCRE.Widestring("\\w");
    string nword = lower_case(word);
    nword = filter(nword, lambda(int c){return regex->match(String.int2char(c));}); 
    if(hyphenation_rules[nword])
      return hyphenation_rules[nword]/"-";
   }
     
#if constant(Public.Tools.Language.Hyphenate)
  if(hyphenator)
  {
    word = hyphenator->hyphenate(word);
werror("hyphenator present\n");
  }
#endif /* have Public.Tools.Language.Hyphenate */
	
    array word_parts = word/"-";
werror("config->unnatural_word_breaks: %O\n", config->unnatural_word_breaks)	;

    if(!(sizeof(word_parts) > 1) && config->unnatural_word_breaks)
    {
werror("splitting unnaturally.\n");
	word_parts = word/"";
    }
	
    return word_parts;
}

void new_paragraph(int|void quad)
{
  werror("quad? %O can justify? %O\n", quad, current_line->can_justify());
  if(!quad && current_line->can_justify()) /* do not quad out */ ; 
  else
  {werror("quadding.\n");
    quad_out();
   werror("yeah!\n");
  }
  werror("quad? %O can justify? %O\n", quad, current_line->can_justify());
  new_line(1);  
}

void make_new_line(int|void newpara)
{
  if(current_line)
    current_line->finalized = 1; // might have to move this further back.
//  if(current_line)
//  werror("*** make_new_line(%f/%f)\n", (float)current_line->lineunits, (float)current_line->linelength);

  if(hanging_punctuation_width != 0.0 && current_line && sizeof(current_line->elements))
  {
    string c = current_line->elements[-1]->character;
    if(!c || !m->is_punctuation(c))
    {
      werror("  adding buffer.\n");
      low_quad_out(hanging_punctuation_width);
    }
    else
    {
//      werror(" NOT adding buffer.\n");
    }
  }
  else
  {
//    werror(" OTHER not adding buffer.\n");
  }
  
  if(current_line && pad_units)
  {
    werror("current_line: %O\n", current_line);
    current_line->lineunits = current_line->lineunits + (pad_units*2);
    low_quad_out((float)pad_units);
    low_quad_out((float)pad_units, 1);
  }

  if(current_line && current_line->js_are_fixed && ((float)current_line->lineunits - (float)current_line->linelength) >= 1.0)
  {
    throw(Error.Generic(sprintf("Line without justifying spaces is short by %f units. Correct by quading out!\n", ((float)current_line->lineunits - (float)current_line->linelength))));
  }
  
//  if(current_line)
//  werror("*** make_new_line(%f/%f)\n", (float)current_line->lineunits, (float)current_line->linelength);
  
  current_line = low_make_new_line();
  
//  werror("made new line.\n");
  if(indent_adjust)
	{
  	float toadd = (float)indent_adjust;
  	if(newpara && toadd > 0)
  	{
  	  werror("indent %O.\n", indent_adjust);
    	if(low_quad_out(toadd) != toadd)
      {
  	    current_line->errors->append(sprintf("Fixed space (%.1f unit) won't fit on line... dropping.\n", toadd));
      }	
    }
    else if(!newpara && toadd < 0)
    {
  	  werror("hanging indent %O.\n", indent_adjust);
      toadd = abs(toadd);
      
    	if(low_quad_out(toadd) != toadd)
      {
  	    current_line->errors->append(sprintf("Fixed space (%.1f unit) won't fit on line... dropping.\n", toadd));
      }	      
    }
	}	
}

// used during new line creation to determine the length of the new line.
// this can vary when setting columnar data when each column may be a different width.
// at this point, lines on page reflects current number of lines on the page, and this function
// should calculate the line length for the *next* line on the page.
int calc_lineunits()
{
  if(!config->widths) return 0;
    
  int col_count = sizeof(config->widths);
  int col;
  if(!sizeof(lines) || sizeof(lines) < (config->page_length[0] * col_count))
  {
    // still on the first page
      col = sizeof(lines) / config->page_length[0];
//      werror("col: %O => %O\n", col, config->widths[col]);
//      throw(Error.Generic("foo\n"));
  }
  else
  { 
    int spl = (sizeof(lines) - (config->page_length[0] * col_count));
    if(sizeof(lines) < (config->page_length[0] * col_count)); // do nothing
    
    else if(sizeof(lines) == (config->page_length[0] * col_count))
    {
      update_page();
    }
    else if(spl>0 && spl % (config->page_length[1] * col_count) == 0)
    {
      update_page();
    }

    // this only comes into play on the last page; so it shouldn't matter that we aren't
    // calculating the current line on the page as if each page could have different lengths.
    int current_page = sizeof(pages);   
    int ps = current_page?1:0;
    if(sizeof(config->page_length) > current_page) ps = current_page;
    
    int psl = sizeof(lines) - ((config->page_length[0] * col_count)); // the number of lines on pages of last page length.
//    werror("psl: %O\n", psl);
    int page = psl / (config->page_length[1] * col_count);
    int lop = psl -  (page * col_count * config->page_length[1]);
//    werror("page: %O\n", page);
//    werror("lop: %O\n", lop);
  werror("page len: %O\n", config->page_length[ps]);
    col = (lop) / config->page_length[ps];
  }
  
  werror("current column number: %d %O\n", col, config->widths);
  config->lineunits = config->widths[col-1];
  return col;
}

Line low_make_new_line(int|void no_increment)
{
	Line l;	
	int col_number;
  col_number = calc_lineunits();

  if(!no_increment)
  {
	  linesonpage++;
	  numline++;
  }
	l = Line(m, s, config + (["lineunits": pad_units?(config->lineunits-(pad_units*2)):config->lineunits]), this);
	l->col_number = col_number;
	l->line_number = numline;
	l->line_on_page = linesonpage;
	return l;
}

object create_styled_sort(string sort, float adjust, void|StyledSort template)
{
  if(template)
    return template->clone(sort);
  else
    return StyledSort(sort, m, config, isitalics, isbold, issmallcaps, adjust, !canHyphenate);
}

// fill out the line according to the justification method (left/right/etc)
void quad_out()
{
//  werror("quad_out()\n");
  current_line->finalized = 1;
  float left = current_line->lineunits - current_line->linelength;
  werror("* have %.1f units left on line.\n", left);

  // we should add a justifying space to a line that has none so that we can be sure
  // that the line will justify properly.
  if(!current_line->linespaces)
  {
    if(line_mode == MODE_CENTER)
    {
      if(left >= (current_line->min_space_units*2))
      {
        current_line->add(JustifyingSpace, 0);
        current_line->add(JustifyingSpace, 1);
      }
    }
    else
    {
      if(left >= current_line->min_space_units)
      {
        if(line_mode == MODE_RIGHT)
          current_line->add(JustifyingSpace, 1);
        else
          current_line->add(JustifyingSpace, 0);
      }
    }
  }
    
  while(!current_line->can_add(left))
  {
//    werror("onoe!\n");
    left --;
//    Tools.throw(Error.Generic, "unable to add %d units because it would cause the line to be overset.\n", left);
  }

  // we've just figured out how many of the requested units will fit _and_ justify, 
  // we don't want to ignore this conclusion.
  // 
  // left = current_line->lineunits - current_line->linelength;

  werror("* %.1f units can be added to %.1f units already on line to give acceptably sized justifying spaces.\n", left, current_line->linelength);

  if(line_mode == MODE_LEFT || line_mode == MODE_JUSTIFY)
  {
 	  low_quad_out(left, 0, 1);
  }
  else if(line_mode == MODE_RIGHT)
  {
    low_quad_out(left, 1, 1);	
  }
  else if(line_mode == MODE_CENTER)
  {
     float l,r;
     l = floor(left/2);
     r = floor((left/2) + (left %2));
	 low_quad_out(r, 0, 1);
	 low_quad_out(l, 1, 1);
  }
}

float low_quad_out(float amount, int|void atbeginning, int|void is_quadding)
{
//  werror("low_quad_out(%f, %d, %f)\n", amount, atbeginning, current_line->linelength);
  
  array toadd = ({});
  float ix;
  float leftover;

  if(!floatp(amount)) amount = (float)amount;
  mapping qspaces;
  if(is_quadding)
    qspaces = quadding_spaces;
  else
    qspaces = spaces;
    
//werror("trying to find a solution for %O with %O\n", (int)floor(amount), qspaces);
  toadd = Monotype.findspace()->simple_find_space((int)floor(amount), qspaces);
  if(!toadd || !sizeof(toadd))
    toadd = Monotype.IterativeSpaceFinder()->findspaces((int)floor(amount), qspaces);
  if(!toadd || !sizeof(toadd))
    toadd = simple_find_space((int)floor(amount), qspaces);

  array list = ({});

    if(sizeof(toadd) && toadd[0] > toadd[-1])
    toadd = reverse(toadd);

  foreach(toadd;;int x)
  {
    if(x == 27) list += ({18,9});
    else list += ({x});
  }

  float tl = (float)Array.sum(list);
  if(tl != amount)
    leftover = amount - tl;

  foreach(list;int z;int i)
  {
    float adjust = 0;
    if(leftover > 0.0)
    {
      if(i < 16)
      {
        if(leftover > 3)
        {
          adjust = 3.0;
          leftover -= 3.0; 
        }
        else
        {
          adjust = leftover, leftover = 0.0;
        }
      }
      else if(i >= 16)
      {
        if(leftover > 1.0)
        {
          adjust = 1.0;
          leftover -= 1.0; 
        }
        else
        {
          adjust = leftover, leftover = 0.0;
        }
      }
    }
    ix+=(i + adjust);
    
    current_line->add(Sort(spaces[i], adjust), atbeginning, 0);	
//   werror("line at %f\n", current_line->linelength);
      if(current_line->is_overset())
      {
        werror("overset. added %.2f, at %f\n", current_line->linelength, (float)ix);
        current_line->remove();
        ix-=(i + adjust);
        if(current_line->can_justify())
          break;
        else
        {
          werror("what's smaller than %d?\n", i);
          array whatsleft = ({});
          // generate an array of available spaces smaller than the one that didn't fit.
          foreach(spaces; mixed u ;)
          {
           if(u < i)
             whatsleft += ({u});
          }
          whatsleft = reverse(sort(whatsleft));
				
        // ok, the plan is to take each space, starting with the biggest and try to add as many
        // of each as possible without going over.
        foreach(whatsleft;;int toadd)
        {
          int cj;
				
          do
          {
            ix+=toadd;
            current_line->add(Sort(spaces[toadd]), atbeginning, 0);	
            cj = current_line->can_justify();
          }
          while(!cj && !current_line->is_overset());
					
          if(current_line->is_overset())
          {
			while(z)
			{
              current_line->remove();
			  z--;
			}
			return 0;
          }
        }
      }
    }
  }
  if((float)ix != (float)amount)
    werror("asked to add %.1f units of space; only added %O.\n", amount, ix);
  return (float)ix;
}

// this an inferior quad-out mechanism. we currently favor
// the algorithm in findspace.pike. left here for historical
// completeness.
array simple_find_space(int amount, mapping spaces)
{
	int left = amount;
	int total = left;

	array toadd = ({});

  foreach(reverse(sort(indices(spaces))); int i; int space)
  {
   while(left > space)
   {
		toadd += ({space});
		left -= space;
   }	
  }

 return toadd;
}

void break_page(int|void newpara)
{
	insert_footer();		
	update_page();  
	insert_header();
}

void update_page()
{
  object p = Page();
	int start;
	if(sizeof(pages))
	  start = pages[-1]->end + 1;
  p->lines = lines[start..];
  p->begin = start;
  p->end = sizeof(lines)-1;
  pages += ({p});	
}

// actually generates the ribbon file from an array of lines of sorts.
// line justifications should already be calculated and provided with each 
// line, however, each sort is checked to make sure its requested width is
// the same as the width of the wedge in the same position. if it's not, 
// we can use various methods (currently consisting only of using the 
// space justification wedges) to "nudge" character to the right width, 
// ensuring justification and world peace.
string generate_ribbon()
{  
	werror("Spaces in matcase: [ %{%d %}]\n", indices(m->spaces));
    werror("*** writing %d lines to the ribbon\n", sizeof(lines));
    String.Buffer buf = String.Buffer();
	
	buf+=sprintf("name: %s\n", config->jobname); 
	buf+=sprintf("face: %s\n", config->matcase->name);
	buf+=sprintf("set: %.2f\n", config->setwidth);
	buf+=sprintf("wedge: %s\n", config->stopbar->name);
	buf+=sprintf("mould: %d\n", config->mould);
	buf+=sprintf("generated: %s\n", Calendar.now()->format_smtp());
	buf+=sprintf("version: %s\n", Monotype.version);
	buf+=sprintf("linelength: %.2f %s\n", config->linelengthp, config->pointsystemname || "");
	if(config->unit_adding)
          buf+=sprintf("unit_adding: %s units\n", (string)config->unit_adding);

	if(config->unit_shift)
          buf+=sprintf("unit_shift: enabled\n");

	buf+=sprintf("\n");
	
	foreach(reverse(lines);; object current_line)
  {
    // if this is the first line and we've opted to make the first line long 
    //  (to kick the caster off,) add an extra space at the beginning.
      if(config->trip_at_end && sizeof(lines) && current_line == lines[0])
      {  		  
     	  string activator = "";
    	  array aspaces = indices(spaces);
    	  int spacesize;
    	  aspaces = sort(aspaces);
        if(aspaces[-1] == 27) //27 isn't a real space.
          aspaces[-1] = 18;
          
    	  if(sizeof(aspaces))
    	    spacesize = aspaces[-1];
    	  if(spacesize)
    	  {
    		  // add at least 18 units of space to the line.
    		  for(int i = 0; i <= 18; i+=spacesize)
          {
            werror("trip space added: %O.\n", spaces[spacesize]);            
    	 		  current_line->add(Sort(spaces[spacesize]), 0, 1);
    	 		  werror("done.\n");
          }
    	  }
    	  else
    	  {
    		  throw(Error.Generic("No spaces in matcase, unable to produce a caster-trip line.\n"));
    	  }		
      }
    
    
    buf += current_line->generate_line();
  }  

  buf+=sprintf("%s %s 1\n", coarse_code, fine_code); // stop the pump, eject the line.

  return (string)buf;
}

// add the current line to the job, if it's justifyable.
void new_line(int|void newpara)
{
  if(!((float)current_line->linelength > 0.0))
  {
    werror("WARNING: new_line() called without any content.\n");
    throw(Error.Generic("foo\n"));
    return 0;
  }    
  if(!current_line->linespaces && abs(current_line->linelength - current_line->lineunits) > 1)
  {
      throw(Error.Generic(sprintf("Off-length line without justifying spaces: need %O units to justify, line has %.2f units. Consider adding a justifying space to line - %s\n", 
		current_line->lineunits, current_line->linelength, (string)current_line)));
  }
  else if(current_line->linespaces && !current_line->can_justify()) 
    throw(Error.Generic(sprintf("Unable to justify line; %f length with %f units, %d spaces, justification code would be: %d/%d, text on line is %s\n", (float)current_line->linelength, (float)current_line->lineunits, current_line->linespaces, current_line->big, current_line->little, string_to_utf8((string)current_line))));

  lines += ({current_line});

//werror("line: %O->%O\n", lines[-1], lines[-1]->elements);

  if(config->page_length && intp(config->page_length) && !(linesonpage%config->page_length))
  {
  	break_page(newpara);
  }
  else
    make_new_line(newpara);
}
