//!  Entry point for migration subsystem
//!
//!

constant MIGRATION_DIR = "db/migration";

//!
object application;

object log = Tools.Logging.get_logger("fins.db.migration");

//!
string migration_dir;

function write_func = Stdio.stdout.write;

protected void create(void|object app)
{
  application = app;  
  
  if(!application)
    throw(Error.Generic("new_migration: no application loaded"));

  migration_dir = Stdio.append_path(application->config->app_dir, MIGRATION_DIR);
}

string make_descriptor(string text)
{ 
  return (string)map(filter((array)Unicode.normalize(lower_case(text), "DK"), `<, 256), lambda(mixed x){if(x<'0') return '_'; else return x;}); 
}

string make_id(object id)
{
  return replace(id->format_time(), ({" ", "-", ":", "."}), ({"", "", "", ""})); 
}

//!
string make_class(string text, object id)
{
  return
  #"inherit Fins.Util.MigrationTask;
  
constant id = \"" + make_id(id) + 
  
  #"\";
constant name=\"" + make_descriptor(text) + 
  #"\";

void up()
{
  
}

void down()
{
  
}
";  
}

void announce(string message, mixed ... args)
{
  int l;
  string m = "== " + message;
  m = sprintf(m, @args);
  l = max(0, 79 - sizeof(m));
  write_func(m + " " + ("="*l) + "\n");
}

array(Fins.Util.MigrationTask) get_migrations()
{
  array f = glob("*.pike", get_dir(migration_dir));

  array migrations = ({});
  
  foreach(f;;string p)
  {
    string taskpath = Stdio.append_path(migration_dir, p);
    object st;
    
    if(!(st = file_stat(taskpath)) || st->isdir)
    {
      werror("skipping %s\n", taskpath);
      continue;
    } 

    program mp = (program)taskpath;
    if(!Program.implements(mp, Fins.Util.MigrationTask))
    {
      werror("program %s doesn't implement MigrationTask.\n", p);
    }
    else
      migrations += ({mp(application, this)});
  }
  
  return migrations;
}

//!
string new_migration(string text, object|void id)
{
  if(!application)
    throw(Error.Generic("new_migration: no application loaded"));
    
  return low_new_migration(text, id, Stdio.append_path(application->config->app_dir, migration_dir));
}

string low_new_migration(string text, object|void id, string dir)
{
  if(!id) id = Calendar.now();
  string c = make_class(text, id);
  string fn = make_id(id) + "_" + make_descriptor(text) + ".pike";
  string afn = Stdio.append_path(dir || getcwd(), fn);
  
  log->debug("Creating new migration in %s.", afn);
  Stdio.write_file(afn, c);
  
  return fn;
}
