#!/usr/local/bin/pike -Mlib

constant my_version = "0.3";
//#define RUN_THREADED

string session_storagedir = "/tmp/scriptrunner_storage";
string logfile_path = "/tmp/scriptrunner.log";
string session_cookie_name = "PSESSIONID";
int session_timeout = 3600;

Stdio.File f;
Stdio.File logfile;
int shutdown = 0;
int requests = 0;
mapping compiled_scripts = ([]);

ScriptRunner.SessionManager session_manager;

int main(int argc, array(string) argv)
{
  int sock;

  write("ScriptRunner " + my_version + " starting.\n");

#ifdef RUN_THREADED
  write("Running in threaded mode.\n");
#endif

  session_manager = ScriptRunner.SessionManager();
  ScriptRunner.SessionStorage s = ScriptRunner.FileSessionStorage();
  s->set_storagedir(session_storagedir);
  session_manager->set_default_timeout(session_timeout);
  session_manager->set_cleaner_interval(session_timeout);
  session_manager->session_storage = ({s});

 
  add_constant("RequestID", ScriptRunner.RequestID);
  add_constant("Session", ScriptRunner.Session);
  add_constant("session_manager", session_manager);
  // ok, our first attempt at session management.

  if(logfile_path)
    logfile=Stdio.File(logfile_path, "rwac");

  f = Stdio.stdin.dup();

#ifdef RUN_THREADED
	for (int i = 0; i < 8; i++) {
		Thread.Thread(request_loop, f->query_fd(), i);
	}
	return (-1);
#else 
	request_loop(f->query_fd(),0);
#endif
}

void request_loop(int sock, int id)
{
        String.Buffer response = String.Buffer();

#ifdef RUN_THREADED
        Thread.Mutex lock;
	Thread.MutexKey key;

        lock = Thread.Mutex();
        key = lock->lock();
#endif
	object request = Public.Web.FCGI.FCGI(sock);
#ifdef RUN_THREADED
        key = 0;
#endif

        do{
		request->accept();
                requests ++;
                object request_id;
                mixed e;

                if(catch(request_id = ScriptRunner.RequestID(request)))
                {
#ifdef RUN_THREADED
                  key = lock->lock();
#endif
                  request->write("Status: 500 Server Error\r\n");
                  request->write("Content-type: text/html\r\n\r\n");
                  request->write("<h1>Error 500: Internal Server Error</h1>");
                  request->write("The server was unable to parse your request.\n");
                  request->finish();
#ifdef RUN_THREADED
                  key = 0;
#endif
                }

                e = catch {

                // do we have a script file passed?
                if(request_id->misc && request_id->misc->path_info && 
                       sizeof(request_id->misc->path_info))
                {
                   object|string s;
                  s = get_script(request_id->misc->path_translated, request_id);
                  if(stringp(s))
                  {
                    request_id->response_write_and_finish("Content-type: text/html\r\n\r\n"
                                   "<h1>Compile Error</h1><pre>" + s + "</pre>");
                    return;
                  }
		  log("running script\n");

                  // do we need to load up a session?
                  if(s["__participates_in_session"] && s["__participates_in_session"] != 0)
                  {
                    if(request_id->cookies && request_id->cookies[session_cookie_name])
                    {
                      ScriptRunner.Session sess = session_manager->get_session(request_id->cookies[session_cookie_name]);
                      request_id->misc->_session = sess;
                      request_id->misc->session_id = sess->id;
                      request_id->misc->session_variables = sess->data;
                    }
                    // if we don't have the session identifier set, we should set one.
                    else 
                    {
                      mapping r = HTTP.set_cookie(session_cookie_name, 
                                                  session_manager->new_sessionid(), time() + session_timeout, request_id);

                      r->status = 302;
                      r->_headers["location"] = request_id->misc->script_name + 
                                                  (request_id->misc->path_info?("/" + request_id->misc->path_info):"") + 
                                                  (request_id->query?("?"+request_id->query):"");

                      request_id->response_write_and_finish( retval_to_response(r));
                      return;
                    }
                  }

                  // the moment of truth!
                  mixed retval;
                  retval = s->parse(request_id);

                  // should we be doing this unconditionally?
                  if(s["__participates_in_session"] && s["__participates_in_session"] != 0)
                  {
                    if(request_id->misc->_session)
                    {
                      // we need to set this explicitly, in case the link was broken.
                      request_id->misc->_session->data = request_id->misc->session_variables;
                      session_manager->set_session(request_id->misc->_session->id, request_id->misc->_session, 
                                                   session_timeout);
                    }
                  }

                  response += retval_to_response(retval);       
                }
                // no, then just print info.
                else
                {
  		  response+="Content-type: text/html\r\n\r\n";
                  response+="<h1>Pike ScriptRunner v" + my_version + "</h1>\n";
		  response+=sprintf("Hello world, this is page (%O) request #%d generated by thread %d\n", request_id->not_query, requests, id);
                  response+="<p><b>Pike Info:</b>\n";
                  response+=sprintf("<pre>\n%s\n</pre>\n", version());
                  response+="<b>Request Info:</b>\n";
		  response+=sprintf("<pre>\nID: %O\n</pre>", 
                    mkmapping(indices(request_id), values(request_id)));
                }

              };

              if(e)
              {
                if(objectp(e))
                  log("got an error: %s\n", e->describe());
                else
                  log("got an error: %O\n", e);
                response+="Content-type: text/html\r\n\r\n";
                response+=sprintf("<h1>\n%s\n</h1>", describe_error(e)); 
                response+=sprintf("<pre>\n%s\n</pre>", describe_backtrace(e)); 
              }

              request_id->response_write_and_finish(response->get());

              log("request finished\n");
                

	} while(!shutdown);
}

string|object get_script(string path, object id)
{
  string code;
  program p;
  Stdio.Stat stat;

  stat = file_stat(path);

  if(!stat)
    error("Script does not exist.\n");

  if((compiled_scripts[path] && compiled_scripts[path][0] == stat->mtime)
         && !id->pragma["no-cache"])
    return compiled_scripts[path][1];

  log("compiling file %s\n", path);

  code = Stdio.read_file(path);

  if(!code)
    error("Script is an empty file.\n");

  if(path[sizeof(path)-4..] == ".psp")
  {
     log("  compiling as a psp\n");
     Web.PikeServerPages.PSPCompiler compiler = Web.PikeServerPages.PSPCompiler();
     compiler->document_root = id->misc->document_root || "/tmp";
     code = compiler->parse_psp(code, path);
  }

  object er = ErrorContainer();

  master()->set_inhibit_compile_errors(er);
  catch(p = compile_string(code));
  master()->set_inhibit_compile_errors(0);
  
  if(!p) // an error must have occurred...
  {
    log("%s", er->get());
    return er->get();
  }

  array ent = ({ stat->mtime, p() });

  compiled_scripts[path] = ent;

  return ent[1];
}

void log(string t, mixed ... args)
{
  if(!logfile) return;

  if(args)
    t = sprintf(t, @args);
  logfile->write(sprintf("[%s] %s", (ctime(time())- "\n"), t));
}




//!
class LowErrorContainer
{
  string d;
  string errors="", warnings="";
  string get()
  {
    return errors;
  }

  //!
  string get_warnings()
  {
    return warnings;
  }

  //!
  void print_warnings(string prefix) {
    if(warnings && strlen(warnings))
      log(prefix+"\n"+warnings);
  }

  //!
  void got_error(string file, int line, string err, int|void is_warning)
  {
    if (file[..sizeof(d)-1] == d) {
      file = file[sizeof(d)..];
    }
    if( is_warning)
      warnings+= sprintf("%s:%s\t%s\n", file, line ? (string) line : "-", err);
    else
      errors += sprintf("%s:%s\t%s\n", file, line ? (string) line : "-", err);
  }

  //!
  void compile_error(string file, int line, string err)
  {
    got_error(file, line, "Error: " + err);
  }
 
  //!
  void compile_warning(string file, int line, string err)
  {  
    got_error(file, line, "Warning: " + err, 1);
  }
      
  //!
  void create()
  {  
    d = getcwd();
    if (sizeof(d) && (d[-1] != '/') && (d[-1] != '\\'))
      d += "/";
  }
}
    
//! @appears ErrorContainer
class ErrorContainer
{
  inherit LowErrorContainer;

  //!
  void compile_error(string file, int line, string err)
  {
//    if( sizeof(compile_error_handlers) )   
//      compile_error_handlers->compile_error( file,line, err );
//    else
      ::compile_error(file,line,err);
  }
    
  //!
  void compile_warning(string file, int line, string err)
  {   
//    if( sizeof(compile_error_handlers) )
//      compile_error_handlers->compile_warning( file,line, err );
//    else
      ::compile_warning(file,line,err);
  }
}

string retval_to_response(mixed retval)
{
   string response="";

              if(!stringp(retval))
                     {
                        if(mappingp(retval))
                        {

                          if(!retval->_headers)
                            retval->_headers = ([]);
                          if(!retval->error)
                            retval->error = 200;
                          if(!retval->type)
                            retval->type = "text/html";

                          response+=sprintf("Status: %d\r\n", retval->error);

                          if(retval->_headers["content-type"]) ; // DO NOTHING!
                          else 
                            response+=sprintf("Content-type: %s\r\n", retval->type);

                          foreach(retval->_headers; string hname; string hvalue)
                            response+=sprintf("%s: %s\r\n", String.capitalize(hname), hvalue);

                          response+="\r\n";

                          // TODO: don't think we can get it all in one call.
                          if(objectp(retval->data))
                            response+=retval->data->read();
                          else if(retval->data)
                            response+=retval->data;

                        } 
                        else error("Invalid return value from parse().\n");
                     }
                     else
                     {
                        response+="Content-type: text/html\r\n\r\n";
                        response+=retval;
                     }

  return response;
}
